<?php

namespace Drupal\webform_content_creator\Plugin\WebformContentCreator\FieldMapping;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_content_creator\Plugin\FieldMappingBase;

/**
 * DateTime field mapping.
 *
 * @WebformContentCreatorFieldMapping(
 *   id = "datetime_mapping",
 *   label = @Translation("DateTime Mapping"),
 *   entity_field_types = {"datetime", "timestamp"},
 * )
 */
class DateTimeMapping extends FieldMappingBase {

  public function getSupportedWebformFields($webform_id) {
    $supported_types = [
      "changed",
      "created",
      "date",
      "datetime",
      "datelist",
      "timestamp",
      "webform_time",
    ];

    return $this->filterWebformFields($webform_id, $supported_types);
  }

  public function mapEntityField(ContentEntityInterface &$content, array $webform_element, array $data = [], FieldDefinitionInterface $field_definition) {
    $fieldId = $field_definition->getName();
    $fieldValue = $data[$fieldId];

    if ($field_definition->getType() == "datetime") {
      $fieldData = $this->convertTimestamp($fieldValue, $field_definition);
      $content->set($fieldId, $fieldData);
    }
  }

  /**
   * Convert timestamp value according to field type.
   *
   * @param int $dateValue
   *   Original datetime value.
   * @param FieldDefinitionInterface $field_definition
   *   Entity field.
   *
   * @return Timestamp
   *   Converted value.
   */
  protected function convertTimestamp($dateValue, FieldDefinitionInterface $field_definition) {
    $dateTime = new DrupalDateTime($dateValue, 'UTC');
    $dateType = $field_definition->getSettings()['datetime_type'];
    if ($dateType === 'datetime') {
      $formatted = \Drupal::service('date.formatter')->format(
        $dateTime->getTimestamp(), 'custom',
        DateTimeItemInterface::DATETIME_STORAGE_FORMAT, 'UTC'
      );
    }
    else {
      $formatted = \Drupal::service('date.formatter')->format(
        $dateTime->getTimestamp(), 'custom', 
        DateTimeItemInterface::DATE_STORAGE_FORMAT, 'UTC'
      );
    }

    return $formatted;
  }

}
