<?php

namespace Drupal\webform_content_creator\Plugin\WebformContentCreator\FieldMapping;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\webform\WebformSubmissionInterface;
use Drupal\webform_content_creator\Plugin\FieldMappingBase;
use Drupal\webform_content_creator\Plugin\FieldMappingInterface;
use Drupal\webform_content_creator\WebformContentCreatorInterface;

/**
 * Text field mapper
 *
 * @WebformContentCreatorFieldMapping(
 *   id = "text_mapping",
 *   label = @Translation("Text Mapping"),
 *   entity_field_types = {"telephone", "text", "text_long", "text_with_summary", "string", "string_long"},
 * )
 */
class TextMapping extends FieldMappingBase {

  public function getSupportedWebformFields($webform_id) {
    $supported_types = array_merge(["string", "string_long", "tel"],
      FieldMappingInterface::WEBFORM_TEXT_ELEMENTS,
      FieldMappingInterface::WEBFORM_OPTIONS_ELEMENTS);

    return $this->filterWebformFields($webform_id, $supported_types);
  }

  public function mapEntityField(ContentEntityInterface &$content, array $webform_element, array $data = [], FieldDefinitionInterface $field_definition) {
    $fieldId = $field_definition->getName();
    $fieldValue = $data[$fieldId];


    $maxLength = $this->checkMaxFieldSizeExceeded($field_definition, $fieldValue);
    if ($maxLength === 0) {
      $content->set($fieldId, $fieldValue);
    }
    else {
      $content->set($fieldId, substr($fieldValue, 0, $maxLength));
    }
  }


  /**
   * Check if field maximum size is exceeded.
   *
   * @param FieldDefinitionInterface $field_definition
   *   Field definition
   * @param string $decValue
   *   Decrypted value.
   *
   * @return int
   *   The max length or length of field, otherwise return 0.
   */
  protected function checkMaxFieldSizeExceeded(FieldDefinitionInterface $field_definition, $decValue = "") {
    $fieldSettings = $field_definition->getSettings();
    if (empty($fieldSettings) || !array_key_exists('max_length', $fieldSettings)) {
      return 0;
    }

    $maxLength = $fieldSettings['max_length'];
    if (empty($maxLength)) {
      return 0;
    }
    if ($maxLength < strlen($decValue)) {
      \Drupal::logger(WebformContentCreatorInterface::WEBFORM_CONTENT_CREATOR)->notice($this->t('Problem: Field max length exceeded (truncated).'));
      return $maxLength;
    }
    return strlen($decValue);
  }

}
