<?php

namespace Drupal\views_lunr_itemsjs\Plugin\views\row;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\row\RowPluginBase;
use Drupal\views_lunr_itemsjs\Entity\lunrItemsjs;
use Drupal\views_lunr_itemsjs\Plugin\views\ExtractFromOptionsTrait;
use Drupal\views_lunr_itemsjs\Plugin\views\field\ExtendedRenderedEntity;

/**
 * Plugin which displays fields as raw data.
 *
 * This class is largely based on the core REST module.
 *
 * @ingroup views_row_plugins
 *
 * @ViewsRow(
 *   id = "lunr_search_index_row",
 *   title = @Translation("LunrItemsjs search index row"),
 *   help = @Translation("Use fields in index."),
 *   display_types = {"lunr_search_index"}
 * )
 */
class LunrSearchIndexRow extends RowPluginBase {

  use ExtractFromOptionsTrait;

  /**
   * {@inheritdoc}
   */
  protected $usesFields = TRUE;

  /**
   * Stores an array of prepared field aliases from options.
   *
   * @var array
   */
  protected array $replacementAliases = [];

  /**
   * Stores an array of options to determine if the raw field output is used.
   *
   * @var array
   */
  protected array $rawOutputFields = [];

  /**
   * Stores an array of options to determine if the field is a filter.
   *
   * @var array
   */
  protected array $filterOutputFields = [];

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL): void {
    parent::init($view, $display, $options);

    if (!empty($this->options['field_options'])) {
      $options = (array) $this->options['field_options'];
      // Prepare a trimmed version of replacement aliases.
      $aliases = static::extractFromOptionsArray('alias', $options);
      $this->replacementAliases = array_filter(array_map('trim', $aliases));
      // Prepare an array of raw output field options.
      $this->rawOutputFields = static::extractFromOptionsArray('raw_output', $options);
      $this->filterOutputFields = static::extractFromOptionsArray('filter', $options);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['field_options'] = ['default' => []];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    $form['field_options'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Field'),
        $this->t('Alias'),
        $this->t('Raw output'),
        $this->t('Filter'),
      ],
      '#empty' => $this->t('You have no fields. Add some to your view.'),
      '#tree' => TRUE,
    ];

    $options = $this->options['field_options'];

    if ($fields = $this->view->display_handler->getOption('fields')) {
      foreach ($fields as $id => $field) {
        // Don't show the field if it has been excluded.
        if (!empty($field['exclude'])) {
          continue;
        }
        $form['field_options'][$id]['field'] = [
          '#markup' => ($field['admin_label'] !== '') ? "$id ({$field['admin_label']})" : $id,
        ];
        $form['field_options'][$id]['alias'] = [
          '#title' => $this->t('Alias for @id', ['@id' => $id]),
          '#title_display' => 'invisible',
          '#type' => 'textfield',
          '#default_value' => $options[$id]['alias'] ?? '',
          '#element_validate' => [[$this, 'validateAliasName']],
        ];
        $form['field_options'][$id]['raw_output'] = [
          '#title' => $this->t('Raw output for @id', ['@id' => $id]),
          '#title_display' => 'invisible',
          '#type' => 'checkbox',
          '#default_value' => $options[$id]['raw_output'] ?? '',
        ];
        $form['field_options'][$id]['filter'] = [
          '#title' => $this->t('@id is a filter', ['@id' => $id]),
          '#title_display' => 'invisible',
          '#type' => 'checkbox',
          '#default_value' => $options[$id]['filter'] ?? '',
        ];
      }
    }
  }

  /**
   * Form element validation handler.
   */
  public function validateAliasName($element, FormStateInterface $form_state): void {
    if (preg_match('@[^A-Za-z0-9_-]+@', $element['#value'])) {
      $form_state->setError($element, $this->t('The machine-readable name must contain only letters, numbers, dashes and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state): void {
    // Collect an array of aliases to validate.
    $aliases = static::extractFromOptionsArray('alias', $form_state->getValue([
      'row_options',
      'field_options',
    ]));

    // If array filter returns empty, no values have been entered. Unique keys
    // should only be validated if we have some.
    if (($filtered = array_filter($aliases)) && (array_unique($filtered) !== $filtered)) {
      $form_state->setErrorByName('aliases', $this->t('All field aliases must be unique'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render($row): array {
    $output = [];
    foreach ($this->view->field as $id => $field) {
      if (empty($field->options['exclude'])) {
        $value = $this->processFieldOutput($field, $row, $id);
        $output[$this->getFieldKeyAlias($id)] = $value;
      }
    }
    $output['_itemsjs_id'] = $row->_entity->uuid();
    return $output;
  }

  protected function processFieldOutput($field, $row, $id) {
    if ($this->isRawOutputField($id)) {
      return $field->getValue($row);
    }
    if ($field instanceof ExtendedRenderedEntity) {
      return $field->render($row)['#markup'];
    }
    if ($this->isFilterOutputField($id)) {
      return $this->filterOutputField($field, $row);
    }
    return $field->advancedRender($row);
  }

  protected function isRawOutputField($id): bool {
    return !empty($this->rawOutputFields[$id]);
  }

  protected function isFilterOutputField($id): bool {
    return !empty($this->filterOutputFields[$id]);
  }

  public function filterOutputField($field, $row) {
    $field->usesOptions = TRUE;
    $field->options['multi_type'] = 'separator';
    $field->options['separator'] = '~~';
    $field->options['delta_limit'] = '50';
    $value = (string) $field->advancedRender($row);
    if (str_contains($value, $field->options['separator'])) {
      $values = explode($field->options['separator'], $value);
      $value = [];
      foreach ($values as $rawValue) {
        $safeName = lunrItemsjs::machineName($rawValue);
        $value += [$safeName => (string) $rawValue];
      }
      return $value;
    }

    $safeName = lunrItemsjs::machineName($value);
    return [$safeName => (string) $value];
  }

  /**
   * Return an alias for a field ID, as set in the options form.
   *
   * @param string $id
   *   The field id to lookup an alias for.
   *
   * @return string
   *   The matches user entered alias, or the original ID if nothing is found.
   */
  public
  function getFieldKeyAlias(string $id
  ): string {
    return $this->replacementAliases[$id] ?? $id;
  }

}
