function lunrSearch(settings) {
  this.documentSourceEndoint = settings.source;

  this.language = settings.language; // Fetch from DrupalSettings?


  this.indexedFields = settings.indexed_fields;
  // Available sort options
  this.sorts = settings.sorts;
  this.indexPrimaryKey = settings.index_primary ?? '_itemsjs_id';

  // Item used for rendering results can be string or template
  this.listItem = settings.list_item;
  this.minChars = parseInt(settings.settings.min_chars) ?? 3;
  this.boostField = (settings.settings.boosted_field !== '0' && settings.settings.boosted_field) ? settings.settings.boosted_field : false;
  this.targetPage = settings.target_page;
  /**
   * Maps the primary key to the original index
   * @type {*[]}
   */
  this.indexMap = [];
  /**
   * Loading state
   * @type {boolean}
   */
  this.isLoading = false;
  /**
   * Search is ready to search
   * @type {boolean}
   */
  this.isReady = false;
  /**
   * Search yielded results
   * @type {boolean}
   */
  this.hasResults = false;
  /**
   * Search has more than this.totalItems results
   * @type {boolean}
   */
  this.hasMoreResults = false;

  /**
   * There is a valid search string
   * @type {boolean}
   */
  this.hasSearch = false;
  /**
   * Has load error?
   * @type {boolean}
   */
  this.hasError = false;

  /**
   * Max items to show
   * @type {number|*}
   */
  this.totalItems = settings.per_page;

  this.totalResults = 0;

  /**
   *  Indexable documents, loaded from the documentSourceEndoint
   * @type {*[]}
   */
  this.documents = [];
  /**
   * Results for this page after a search has been performed.
   * @type { {} }
   */
  this.searchResults = {};

  /**
   * The query to search for.
   * @type {string}
   */
  this.searchQuery = '';

  /**
   * Map of the search results
   * @type {Map<any, any>}
   */
  this.searchResultMap = new Map();
  /**
   * Bind for fully indexed Lunr
   * @type {lunr|{}}
   */
  this.lunr = {};
}

/**
 * Callback after the data is loaded.
 */
lunrSearch.prototype.initLunrSearch = function () {

  this.documents.map((document, index) => {
    this.indexMap[document[this.indexPrimaryKey]] = index;
  });

  const _this = this;
  this.lunr = new lunr(function () {
    this.pipeline.remove(lunr.stemmer)
    this.ref(_this.indexPrimaryKey);
    Object.keys(_this.indexedFields).map(field => {
      this.field(field);
    });
    _this.documents.forEach(function (document) {
      this.add(document);
    }, this);
  });
  this.isLoading = false;
  this.isReady = true;
  this.search();
};

lunrSearch.prototype.reset = function () {
  this.searchResults = [];
  this.totalResults = 0;
  this.searchResultMap = new Map();
  this.updateStatus();
}

lunrSearch.prototype.updateStatus = function () {
  this.hasResults = (this.totalResults > 0);
  this.hasSearch = (this.searchQuery.length >= this.minChars)
  this.hasMoreResults = (this.totalResults > 0 && this.totalResults > this.totalItems);
  if (!this.hasResults) {
    this.searchResults = {}
  }
};

lunrSearch.prototype.getTargetPage = function () {
  if (this.targetPage) {
    const targetUrl = new URL(this.targetPage);
    if (this.searchQuery) {
      const targetQueryString = new URLSearchParams({search: this.searchQuery}).toString();
      targetUrl.search = targetQueryString;
    }
    return targetUrl.toString();
  }
  return false;
};

lunrSearch.prototype.gotoPage = function () {
  const targetUrl = this.getTargetPage();
  if (targetUrl) {
    location.href = targetUrl;
  }
};


lunrSearch.prototype.search = function () {
  if (this.searchQuery.length >= this.minChars) {
    const words = this.searchQuery
      .split(' ')
      .map(word => word.trim())
      .filter(word => word.length > 0);

    const searchQuery =
      words.map(word => `*${word}*`)
        .join(' ');

    this.mapSearchResults(this.lunr.search(searchQuery));
    this.searchBoostField(words);

    if (this.postFilter && this.postFilter instanceof Function) {
      this.postFilter();
    }
    this.totalResults = this.searchResultMap.size;
    if (this.totalResults > 0) {
      this.mapSearchResultAsOutput();
    }
  }
  else {
    this.totalResults = 0;
    this.searchResults = [];
  }
  this.updateStatus();
};

/**
 * Binds to input @search. event, and resets
 * only if there is no search query.
 * No need to search twice if you have a
 * @input. event as well.
 */
lunrSearch.prototype.resetSearch = function () {
  if (this.searchQuery.length === 0) {
    this.reset();
  }
}
/**
 *
 * @param words
 */
lunrSearch.prototype.searchBoostField = function (words) {
  if (this.boostField) {
    const boostField = this.boostField;
    const boostQuery = words
      .map(word => `+${boostField}:${word}`)
      .join(' ');
    const boostResults = this.lunr.search(boostQuery);
    boostResults.forEach(result => {
      // boost search result score if
      if (this.searchResultMap.has(result.ref)) {
        let score = this.searchResultMap.get(result.ref);
        score = score + result.score;
        this.searchResultMap.set(result.ref, score);
      }
    });
  }
};

/**
 * Maps the search results from Lunr's search response to a custom output
 * format. that can be used as output in the template.
 */

lunrSearch.prototype.mapSearchResultAsOutput = function () {
  let resultsArray = Array.from(this.searchResultMap.keys());
  if (this.totalResults > this.totalItems) {
    resultsArray = resultsArray.slice(0, this.totalItems);
  }

  this.searchResults = resultsArray.map((refId) => {
    return {
      listItem: this.documents[this.indexMap[refId]][this.listItem],
      ref: refId
    };
  });
}

/**
 * Create mapable object of the search results.
 * For internal use.
 * @param searchResults
 */
lunrSearch.prototype.mapSearchResults = function (searchResults) {
  this.searchResultMap = new Map();
  searchResults.forEach(result => {
    this.searchResultMap.set(result.ref, result.score);
  });
};

/**
 * Load the data from the endpoint.
 */
lunrSearch.prototype.loadData = async function () {
  this.isLoading = true;
  const self = this;
  const xhr = new XMLHttpRequest();
  xhr.open('GET', this.documentSourceEndoint);
  xhr.responseType = 'json';
  xhr.onload = function () {
    if (this.readyState === 4) {
      self.documents = this.response;
      self.initLunrSearch();
    }
    else {
      self.isLoading = false;
      self.hasError = true;
    }
  };
  xhr.send();
};


document.addEventListener('alpine:init', () => {
  Alpine.data('lunrAutocomplete', () => ({
    lunrSearch: null,
    init() {
      // Create the lunrSearch instance
      this.lunrSearch = new lunrSearch(drupalSettings.lunrItemsjsSettings);
      // Call loadData() in the background
      this.lunrSearch.loadData()
        .catch((error) => {
          console.error('Data loading failed:', error);
        });
    },
  }));
});

