<?php

namespace Drupal\views_lunr_itemsjs\Entity;

use Drupal\Component\Utility\Html;
use Drupal\Core\Config\Entity\ConfigEntityBase;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\views_lunr_itemsjs\lunrItemsConfigInterface;
use stdClass;

/**
 * Defines the search configuration entity type.
 *
 * @ConfigEntityType(
 *   id = "lunr_itemsjs",
 *   label = @Translation("Lunr-ItemsJS Configuration"),
 *   label_collection = @Translation("Lunr-ItemsJS Configurations"),
 *   label_singular = @Translation("Lunr-ItemsJS Configuration"),
 *   label_plural = @Translation("Lunr-ItemsJS Configurations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count Configuration",
 *     plural = "@count Configurations",
 *   ),
 *   handlers = {
 *   "list_builder" ="Drupal\views_lunr_itemsjs\LunrItemsjsConfigListBuilder",
 *   "form" = {
 *     "add" = "Drupal\views_lunr_itemsjs\Form\LunrItemsjsConfigurationForm",
 *     "edit" = "Drupal\views_lunr_itemsjs\Form\LunrItemsjsConfigurationForm",
 *     "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *    }
 *   },
 *   config_prefix = "lunr_itemsjs",
 *   admin_permission = "administer lunr_itemsjs config",
 *   links = {
 *     "collection" = "/admin/structure/lunritemsjs",
 *     "add-form" = "/admin/structure/lunritemsjs/add",
 *     "edit-form" = "/admin/structure/lunritemsjs/{lunr_itemsjs}/edit",
 *     "delete-form" = "/admin/structure/lunritemsjs/{lunr_itemsjs}/delete"
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "view",
 *     "display",
 *     "filters",
 *     "indexed_fields",
 *     "sorts",
 *     "per_page",
 *     "list_item",
 *     "visible_links",
 *     "settings",
 *   }
 * )
 */
class lunrItemsjs extends ConfigEntityBase implements lunrItemsConfigInterface {

  protected string $id;

  protected string $label;

  protected ?string $description = NULL;

  /**
   * various settings that don't belong in a
   * variable of their own.
   */
  protected ?array $settings = [];

  /**
   * The view that builds the index
   */
  protected ?string $view = NULL;

  /**
   * Type of configuration, currently
   * 'autocomplete'
   * 'overview'
   * are implemented.
   *
   */
  protected ?string $display = NULL;

  /**
   * The available facets if applicable
   */
  protected ?array $filters = [];

  /**
   * The available sorters if applicable
   */
  protected ?array $sorts = [];

  /**
   * The fields used for Lunr index
   */
  protected ?array $indexed_fields = [];

  /**
   * Limit items per page / dropdown
   */

  protected ?int $per_page = 25;

  /**
   * the total numer of visible pagination links.
   */

  protected ?int $visible_links = 5;

  /**
   * item used for building the list;
   */
  protected ?string $list_item = NULL;

  public function getViewID(): ?string {
    return $this->view;
  }

  public function getSettings(): ?array {
    return $this->settings;
  }

  public function getDisplayType(): ?string {
    return $this->display;
  }

  public function getListItem(): ?string {
    return $this->list_item;
  }

  public function getFilters(): ?array {
    return $this->filters;
  }

  public function getSorts(): ?array {
    return $this->sorts;
  }

  public function getIndexedFields(): ?array {
    $return = [];
    foreach ($this->indexed_fields as $field => $active) {
      if ((bool) $active) {
        $return[$field] = $active;
      }
    }
    return $return;
  }

  public function getItemsPerPage(): ?string {
    return $this->per_page;
  }

  public function getVisibleLinks(): ?string {
    return $this->visible_links;
  }

  /**
   * @param string $setting
   * Get a single setting
   *
   */
  public function getSetting(string $setting) {
    if (!empty($this->settings) && isset($this->settings[$setting])) {
      return $this->settings[$setting];
    }
  }

  public function buildBlockSorts(): array {
    $config = [];
    $defaultSortFound = FALSE;
    $defaultSort = NULL;
    foreach ($this->sorts as $index => $sort) {
      $delta = self::machineName($sort['label']);
      $isDefault = (bool) $sort['default_sort'];
      if ($isDefault && !$defaultSortFound) {
        $defaultSort = $delta;
        $defaultSortFound = TRUE;
      }
      $sortFields = [];
      $sortOrders = [];
      $sortConfig = explode(",", $sort['config']);

      foreach ($sortConfig as $configPair){
       [$field,$order] = explode(":", $configPair);
       $sortFields[] = trim($field);
       $sortOrders[] = trim($order);
       }
      $config['configs'][$delta] = [
        'title' => t($sort['label']),
        'title_slug' => t($delta),
        'config' => ['field' => $sortFields,'order' => $sortOrders],
      ];
    }
    // todo add first as default sort!
    $config['default_sort'] = $defaultSort;
    return $config;
  }

  /**
   * TODO Rebuild this when adjusting the filters@
   *
   * @return array
   */
  public function buildBlockFilters(): array {
    $config = [];
    foreach ($this->filters as $key => $filter) {
      $config[$filter['field']] = [
        'title' => t($filter['label']),
        'title_slug' => t(self::machineName($filter['label'])),
        'field' => $filter['field'],
        'conjunction' => (bool) $filter['conjunction'],
        'size' => $filter['size'],
        'order' => $filter['order'],
        'sort' => $filter['sort'],
        'chosen_filters_on_top' => (bool) $filter['selected_first'],
        'hide_zero_doc_count' => (bool) $filter['hide_zero'],
        'show_facet_stats' => FALSE,
        // Not sure if this needs to be configurable
      ];
    }
    return $config;
  }

  /**
   * Get the actual source view
   *
   * @return \Drupal\views\ViewExecutable|false
   */
  public function getViewExecutable() {
    /** @var \Drupal\views\ViewEntityInterface $view */
    [$viewID, $viewDisplay] = explode('.', $this->getViewID());
    $view = $this->getViewsStorage()->load($viewID);
    if ($view) {
      $executable = $view->getExecutable();
      if ($executable->setDisplay($viewDisplay)) {
        return $executable;
      }
    }
    return FALSE;
  }

  public function calculateDependencies() {
    parent::calculateDependencies();
    $this->addDependency('module', 'views');
    if ($this->getViewId()) {
      [$viewID, $viewDisplay] = explode('.', $this->getViewID());
      $this->addDependency('config', "views.view.{$viewID}");
    }
    return $this;
  }

  /**
   * Transform a string into a machine friendly name
   */
  public static function machineName(string $name): string {
    $name = HTML::decodeEntities($name);
    if (!preg_match('/^[a-z0-9\_]+$/', $name)) {
      $unknown_character = '_';
      // Transliterate and sanitize the destination filename.
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
      $machine_name = \Drupal::transliteration()
        ->transliterate($name, $langcode, $unknown_character);
      $machine_name = mb_strtolower(trim($machine_name));
      $machine_name = trim(preg_replace('/[^a-z0-9\_]+/', $unknown_character, $machine_name), $unknown_character);
      $name = $machine_name;
    }
    return $name;
  }

  protected function getViewsStorage(): EntityStorageInterface {
    return \Drupal::entityTypeManager()->getStorage('view');
  }

}
