<?php

namespace Drupal\views_lunr_itemsjs\Plugin\views\style;

use Drupal;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\SerializerInterface;

// TODO, Set default views caching to disabled without altering the options!
// Or move turbo cache into cache options instead.

/**
 * The style plugin for serialized output formats.
 *
 * This class is largely based on the core REST module.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "lunr_search_index_json",
 *   title = @Translation("LunrItemsjs search index JSON"),
 *   help = @Translation("Serializes views row data to JSON."),
 *   display_types = {"lunr_search_index"}
 * )
 */
class LunrSearchIndexJson extends StylePluginBase implements CacheableDependencyInterface {

  /**
   * {@inheritdoc}
   */
  protected $usesRowPlugin = TRUE;

  /**
   * {@inheritdoc}
   */
  protected $usesGrouping = FALSE;

  /**
   * The serializer which serializes the views result.
   *
   * @var \Symfony\Component\Serializer\Serializer
   */
  protected $serializer;

  /**
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $renderCache;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('serializer'),
      $container->get('cache.render')
    );
  }

  /**
   * Constructs a Plugin object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, SerializerInterface $serializer, CacheBackendInterface $render_cache) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->serializer = $serializer;
    $this->renderCache = $render_cache;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $rows = [];
    $results = [];

    $turboCacheEnabled = (bool) $this->options['turbo_cache'];
    // In preview mode, we never ever output cached data!
    if (property_exists($this->view, 'live_preview') && $this->view->live_preview) {
      $turboCacheEnabled = FALSE;
    }

    $language = Drupal::languageManager()->getCurrentLanguage()->getId();
    $cacheId = 'views:' . $language . ':' . $this->view->id() . '-' . $this->view->current_display;

    foreach ($this->view->result as $result) {
      $entity = $result->_entity;
      $eid = $entity->id();
      if (!$results[$eid]) {
        $results[$eid] = $result;
      }
    }
    if ($turboCacheEnabled) {
      $viewCache = $this->renderCache->get($cacheId);
      if ($viewCache) {
        return $viewCache->data;
      }
    }

    foreach ($results as $row_index => $row) {
      $this->view->row_index = $row_index;

      $output = $this->view->rowPlugin->render($row);
      $uuid = $output['_itemsjs_id'];
      // UUID is assumed to be unique in single overview!
      $rows[$uuid] = $output;
    }

    unset($this->view->row_index);
    $data = $this->serializer->serialize(array_values($rows), 'json');
    if ($turboCacheEnabled) {
      $this->renderCache->set($cacheId, $data, $this->getCacheMaxAge(), $this->getCacheTags());
    }
    return $data;
  }

  /**
   * {@inheritdoc}
   * This has no influence at this moment
   */
  public function getCacheContexts() {
    return ['request_format'];
  }

  /**
   * {@inheritdoc}
   *
   */
  public function getCacheMaxAge() {
    return Cache::PERMANENT;
  }

  /**
   * {@inheritdoc}
   * Turbo cache this result till the end of time.
   */
  public function getCacheTags() {
    $cacheTags = array_filter($this->view->getCacheTags(), static function($item) {
      return $item !== 'node_list';
    });

    $cacheTags[] = $this->getCacheTagForCurrentViewDisplay();
    return array_values($cacheTags);
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = parent::calculateDependencies();
    $dependencies['module'][] = 'serialization';
    return $dependencies;
  }

  protected function defineOptions(): array {
    $options['turbo_cache']['default'] = FALSE;
    return $options;
  }

  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    $form['turbo_cache'] = [
      '#title' => $this->t('Enable turbo cache'),
      '#type' => 'checkbox',
      '#default_value' => $this->options['turbo_cache'],
    ];
    $form['turbo_cache_container'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="style_options[turbo_cache]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['turbo_cache_container']['turbo_cache_description'] = [
      '#type' => 'markup',
      '#markup' => $this->t('<pre>Implement : Cache::invalidateTags(["%cache_id"]);</pre>',
        ['%cache_id' => $this->getCacheTagForCurrentViewDisplay()]),
    ];
  }

  public function getCacheTagForCurrentViewDisplay() {
    return sprintf("lunr_view:%s-%s", $this->view->id(), $this->view->current_display);
  }

}
