<?php

namespace Drupal\views_lunr_itemsjs\Plugin\views\display;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\Render\RendererInterface;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\display\ResponseDisplayPluginInterface;
use Drupal\views\Render\ViewsRenderPipelineMarkup;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The plugin that handles LunrItemsjs search indexes.
 *
 * This class is largely based on the core REST module.
 *
 * @ingroup views_display_plugins
 *
 * @ViewsDisplay(
 *   id = "lunr_search_index",
 *   title = @Translation("LunrItemsjs search index"),
 *   help = @Translation("Create a LunrItemsjs search index."),
 *   admin = @Translation("LunrItemsjs search index"),
 *   contextual_links_locations = {""},
 *   lunr_search_index = TRUE
 * )
 */
class LunrSearchIndex extends DisplayPluginBase implements ResponseDisplayPluginInterface {

  /**
   * {@inheritdoc}
   */
  protected $usesAJAX = FALSE;

  /**
   * {@inheritdoc}
   */
  protected $usesPager = FALSE;

  /**
   * {@inheritdoc}
   */
  protected $usesMore = FALSE;

  /**
   * {@inheritdoc}
   */
  protected $usesAreas = FALSE;

  /**
   * {@inheritdoc}
   */
  protected $usesOptions = FALSE;


  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a LunrSearchIndex object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state key value store.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, StateInterface $state, RendererInterface $renderer) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $state);
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('state'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return 'lunr_search_index';
  }

  /**
   * {@inheritdoc}
   */
  public function usesExposed() {
    return FALSE;
  }


  /**
   * {@inheritdoc}
   */
  public function displaysExposed() {
    return FALSE;
  }

  public function usesPager() {
    return FALSE;
  }

  public function isPagerEnabled() {
    return FALSE;
  }


  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    $options['style']['contains']['type']['default'] = 'lunr_search_index_json';
    $options['row']['contains']['type']['default'] = 'lunr_search_index_row';
    $options['defaults']['default']['style'] = FALSE;
    $options['defaults']['default']['row'] = FALSE;
    $options['defaults']['default']['header'] = FALSE;
    $options['defaults']['default']['footer'] = FALSE;



    $options['pager']['contains']['type']['default'] = 'none';
    //$options['defaults']['default']['sorts'] = FALSE;

    // Remove css/exposed form settings.
    unset($options['exposed_form'], $options['exposed_block'], $options['css_class']);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function optionsSummary(&$categories, &$options) {
    parent::optionsSummary($categories, $options);

    unset($categories['page'], $categories['exposed'], $options['show_admin_links'], $options['analyze-theme']);
    unset($options['exposed_form'], $options['exposed_block'], $options['css_class']);
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    parent::execute();
    return $this->view->render();
  }

  public function preview() {
    return $this->view->render();
  }

  /**
   * {@inheritdoc}
   * @throws \JsonException
   */
  public function render() {

    $build = [];
    $build['#markup'] = $this->renderer->executeInRenderContext(new RenderContext(), function () {
      return $this->view->style_plugin->render();
    });

    // Encode and wrap the output in a pre tag if this is for a live preview.
    if (property_exists($this->view, 'live_preview') && $this->view->live_preview) {
      $build['#prefix'] = '<pre>';
      $build['#plain_text'] = json_encode($build['#markup'], JSON_THROW_ON_ERROR | JSON_PRETTY_PRINT);
      $build['#suffix'] = '</pre>';
      unset($build['#markup']);
    }
    else {
      $build['#markup'] = json_decode($build['#markup'], TRUE, 512, JSON_THROW_ON_ERROR);
    }
    $this->applyDisplayCacheabilityMetadata($build);
    return $build;
  }

  /**
    * Override cache, 100% determined by the view style
   *  @see lunr_search_index_json
    */
  protected function applyDisplayCacheabilityMetadata(array &$element) {
    (new CacheableMetadata())
      ->setCacheTags($this->view->getStyle()->getCacheTags())
      ->setCacheContexts($this->display['cache_metadata']['contexts'] ?? [])
      ->setCacheMaxAge( Cache::PERMANENT)
      ->applyTo($element);
  }
  /**
   * {@inheritdoc}
   * todo: is this actually used? Looks like it never is hit all?
   */
  public static function buildResponse($view_id, $display_id, array $args = []) {
    $build = static::buildBasicRenderable($view_id, $display_id, $args);

    // Set up an empty response so headers can be added as needed during views
    // rendering and processing.
    $response = new CacheableResponse('', 200);
    $build['#response'] = $response;

    /** @var \Drupal\Core\Render\RendererInterface $renderer */
    $renderer = \Drupal::service('renderer');

    $output = (string) $renderer->renderRoot($build);

    $response->setContent($output);
    $cache_metadata = CacheableMetadata::createFromRenderArray($build);

    $response->addCacheableDependency($cache_metadata);

    $response->headers->set('Content-type', 'application/json');


    return $response;
  }

}
