<?php

namespace Drupal\views_lunr_itemsjs\Plugin\rest\resource;


use Drupal\Core\Config\Entity\ConfigEntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\rest\Plugin\ResourceBase;
use Drupal\rest\ResourceResponse;
use Drupal\views_lunr_itemsjs\Entity\lunrItemsjs;
use Psr\Log\LoggerInterface;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

/**
 * Represents LunrItemsjs Views as consumable output
 *
 * @RestResource (
 *   id = "views_lunr_itemsjs_rest_resource",
 *   label = @Translation("Lunr Itemsjs Resource"),
 *   uri_paths = {
 *     "canonical" = "/api/lunr-itemsjs-resource/{resource}"
 *   }
 * )
 *
 */
class RestLunrItemsjsResource extends ResourceBase {


  protected ConfigEntityStorageInterface $configStorage;

  protected ?lunrItemsjs $lunrItemsConfig;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array                      $configuration,
                               $plugin_id,
                               $plugin_definition,
    array                      $serializer_formats,
    LoggerInterface            $logger,
    EntityTypeManagerInterface $entityTypeManager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer_formats, $logger);
    $this->configStorage = $entityTypeManager->getStorage('lunr_itemsjs');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->getParameter('serializer.formats'),
      $container->get('logger.factory')->get('rest'),
      $container->get('entity_type.manager')
    );
  }


  public function availableMethods() {
    return ['GET'];
  }

  public function get(string $resource): ResourceResponse {

    if (!$this->isResourceDefined($resource)) {
      throw new NotFoundHttpException("$resource not found.");
    }
    $responseData = [];
    $responseData['config'] = $this->buildConfiguration();
    $responseData['documents'] = $this->buildDocuments();
    return new ResourceResponse($responseData);
  }


  /**
   * {@inheritdoc}
   */
  protected function getBaseRoute($canonical_path, $method) {
    $route = parent::getBaseRoute($canonical_path, $method);
    $route->setRequirement('resource', '\w+');
    return $route;
  }

  protected function isResourceDefined(string $resource): bool {
    $resource = $this->configStorage->load($resource);
    if (!$resource) {
      return FALSE;
    }
    $this->lunrItemsConfig = $resource;
    return TRUE;
  }


  protected function buildFilters() {
    $filterConfig = [];
    foreach ($this->lunrItemsConfig->getFilters() as $key => $filter) {
      $filterConfig[$filter['field']] = [
        'title' => $filter['label'],
        'title_slug' => lunrItemsjs::machineName($filter['label']),
        'field' => $filter['field'],
        'conjunction' => (bool) $filter['conjunction'],
        'size' => $filter['size'],
        'order' => $filter['order'],
        'sort' => $filter['sort'],
        'selected_first' => (bool) $filter['selected_first'],
        'hide_zero_doc_count' => (bool) $filter['hide_zero'],
        'show_facet_stats' => FALSE,
        // Not sure if this needs to be configurable
      ];
    }
    return $filterConfig;
  }

  protected function buildSorts() {
    $sortConfig = [];
    foreach ($this->lunrItemsConfig->getSorts() as $index => $sort) {
      $delta = lunrItemsjs::machineName("{$sort['field']}-{$index}");
      $sortConfig[$delta] = [
        'title' => $sort['label'],
        'title_slug' => lunrItemsjs::machineName($sort['label']),
        'field' => $sort['field'],
        'order' => $sort['order'],
      ];
    }
    return $sortConfig;

  }

  protected function buildConfiguration(): array {
    $config = [];
    $config['filters'] = $this->buildFilters();
    $config['sort'] = $this->buildSorts();
    $config['list_item'] = $this->lunrItemsConfig->getListItem();
    $config['per_page'] = $this->lunrItemsConfig->getItemsPerPage();
    $config['pagination_visible_links'] = $this->lunrItemsConfig->getVisibleLinks();
    $config['indexed_fields'] = $this->lunrItemsConfig->getIndexedFields();
    $config['search_options'] = $this->lunrItemsConfig->getSettings();
    return $config;
  }

  protected function buildDocuments(): array {
    $view = $this->lunrItemsConfig->getViewExecutable();
    $results = $view->getDisplay()->execute();
    return $results && $results['#markup'] ? $results['#markup'] : [];
  }

}
