const lunrItemsjs = function (settings) {

  this.documentSourceEndoint = settings.source;
  this.pageUrl = new URL(window.location.href);

  this.language = settings.language; // Fetch from DrupalSettings?
  this.boostField = (settings.settings.boosted_field !== '0' && settings.settings.boosted_field) ? settings.settings.boosted_field : false;
  // The Facet filters
  this.filters = settings.filters;
  // Lunr Searchable fields
  this.indexedFields = settings.indexed_fields;
  // Available sort options
  this.sorts = {};
  Object.keys(settings.sorts.configs).map(key => {
    this.sorts[key] = settings.sorts.configs[key].config;
  });
  this.activeSort = settings.sorts.default_sort;
  // Primary key of indexed items, hardcoded in the view.
  this.indexPrimaryKey = settings.index_primary ?? '_itemsjs_id';

  // Item used for rendering results can be string or template
  this.listItem = settings.list_item;

  /**
   * Lookup table for internal usage.
   * @type {*[]}
   */
  this.lookups = [];

  this.activeFilters = [];
  this.totalResults = -1;

  /**
   * Loading state
   * @type {boolean}
   */
  this.isLoading = false;
  /**
   * Has load error?
   * @type {boolean}
   */
  this.hasError = false;

  /**
   * Items per page of pagination is used
   * @type {number|*|string}
   */
  this.itemsPerPage = settings?.per_page ?? '-1';
  /**
   * Number of visible links for pagination
   * @type {number}
   */
  this.visibleLinks = parseInt(settings?.pagination_visble_links ?? 5);

  /**
   * Minimal tokens for a keyword search
   * @type {*|number}
   */
  this.minChars = parseInt(settings.settings.min_chars) ?? 3;
  /**
   *  Indexable documents, loaded from the documentSourceEndpoint
   * @type {*[]}
   */
  this.documents = [];
  /**
   * Results for this page after a search has been performed.
   * @type { {} }
   */
  this.searchResults = {};

  this.hasResults = false;
  /**
   * State of the current 'search'
   * @type {{filters: *[], search: string, currentPage: number}}
   */
  this.searchQuery = {
    filters: [],
    search: '',
    currentPage: 1,
  };

  this.pageUrl = new URL(window.location.href);

  this.itemsjsConfiguration = {
    sortings: this.sorts,
    aggregations: this.filters,
    searchableFields: Object.keys(this.indexedFields),
    native_search_enabled: false,
    custom_id_field: this.indexPrimaryKey,
  };
  // Helper structure to make pagination easier.

  this.pagination = {
    range: 0,
    rangeStart: 0,
    rangeEnd: 0,
    currentPage: 1,
    FirstPage: 1,
    lastPage: Math.ceil(this.totalResults / this.itemsPerPage),
    totalResults: this.totalResults,
  };

  /**
   * Configuration for ItemsJS is rather static
   * @type {{searchableFields: string[], sortings, native_search_enabled:
   *   boolean, aggregations, custom_id_field: (*|string)}}
   */

  /**
   * Bind for fully indexed Lunr
   * @type {lunr|{}}
   */
  this.lunr = {};
  /**
   * Bind for fully indexed itemsjs
   * @type {itemsjs|{}}
   */
  this.itemsjs = {}; // Will hold the ItemsJS system
};

lunrItemsjs.prototype.initItemsJS = function () {

  this.itemsjs = itemsjs(this.documents, this.itemsjsConfiguration);
};

lunrItemsjs.prototype.initLunrSearch = function () {
  const _this = this;
  this.lunr = new lunr(function () {
    /**
     * The primary key for this index is `_itemsjs_id` as defined in the view
     * Make sure if you change that in a view, you change that in the
     * drupalSettings as well!
     *
     * Since users can add a custom field called ID it's set to this less
     * common guessable primary key by default.
     *
     * @see \Drupal\views_lunr_itemsjs\Plugin\search_api\backend\LunrBackend::indexItems
     */
    this.ref(_this.indexPrimaryKey);
    Object.keys(_this.indexedFields).map(field => {
      this.field(field);
    });
    _this.documents.forEach(function (document) {
      this.add(document);
    }, this);
  });
};

lunrItemsjs.prototype.initView = function () {
  this.preProcessDocuments();
  this.initLunrSearch();
  this.initItemsJS();
  this.initSlugLookups();
  this.emptySearchQuery();
  this.executeSearch();
  this.updateSearchQueryFromUrl();
  this.updatePagination();
  // If the URL has a searchString bind that to the searchQuery and redo search
  this.isLoading = false;
};

/**
 * Returns a single active filter if available.
 * @param filterName
 * @returns {*|boolean}
 */
lunrItemsjs.prototype.getActiveFilter = function (filterName) {
  return (this.activeFilters.hasOwnProperty(filterName)) ? this.activeFilters[filterName] : false;
};

/**
 * The click target for a reset button.
 */
lunrItemsjs.prototype.reset = function () {
  this.emptySearchQuery();
  this.search();
};

/**
 * Helper for pagination.
 * @param page
 */
lunrItemsjs.prototype.gotoPage = function (page) {
  this.searchQuery.currentPage = page;
  this.search();
};
/**
 * Update Pagination records after search has been executed.
 */
lunrItemsjs.prototype.updatePagination = function () {
  const pageRange = this.visibleLinks;

  const lastPage = Math.ceil(this.totalResults / this.itemsPerPage);
  const firstPage = 1;
  const currentPage = this.currentPage;
  const offset = Math.floor(pageRange / 2);

  let rangeEnd = this.currentPage + offset;
  rangeEnd = rangeEnd < pageRange ? pageRange : rangeEnd;
  rangeEnd = rangeEnd > lastPage ? lastPage : rangeEnd;

  let rangeStart = currentPage - offset;
  rangeStart = lastPage < rangeStart + pageRange ? lastPage - pageRange + 1 : rangeStart;
  rangeStart = rangeStart < firstPage ? firstPage : rangeStart;

  const preElipse = (rangeEnd > pageRange && rangeStart < currentPage);
  const postElipse = (currentPage < rangeEnd && lastPage > rangeEnd);

  let rangeItems = {};
  for (let range = rangeStart; range <= rangeEnd; range++) {
    rangeItems[range] = { 'active': (range === this.currentPage) };
  }

  this.pagination = {
    rangeItems: rangeItems,
    rangeStart: rangeStart,
    rangeEnd: rangeEnd,
    rangeOffset: offset,
    currentPage: currentPage,
    firstPage: 1,
    lastPage: lastPage,
    totalResults: this.totalResults,
    itemsPerPage: this.itemsPerPage,
    preElipse: preElipse,
    postElipse: postElipse,
  };
};

/**
 * Update this.searchQuery fom pageUrl.searchParams
 */
lunrItemsjs.prototype.updateSearchQueryFromUrl = function () {
  let inversedLookup = [];
  for (const [key, value] of Object.entries(this.lookups['slugs'])) {
    inversedLookup[value] = key;
  }
  const searchParams = Array.from(this.pageUrl.searchParams);
  if (searchParams.length > 0) {
    searchParams.forEach(filterPairs => {
      const key = filterPairs[0];
      const filter = inversedLookup[key];
      if (this.searchQuery.filters[filter]) {
        this.searchQuery.filters[filter] = filterPairs[1].split('.');
      } else {
        this.searchQuery[filter] = filterPairs[1].replaceAll('+', ' ');
      }
    });
    const hasPage = this.pageUrl.searchParams.get('page');
    if (hasPage) {
      const lastPage = Math.ceil(this.totalResults / this.itemsPerPage);
      if (hasPage <= lastPage) {
        this.searchQuery.currentPage = parseInt(hasPage);
      }
    }
    this.executeSearch();
  }
};
/**
 * Update the page url from this.searchQuery content
 */
lunrItemsjs.prototype.updateURLfromSearchQuery = function () {
  const filterQueryString = {};
  Object.keys(this.searchQuery).map(filterName => {
    const slug = this.lookups['slugs'][filterName];
    switch (filterName) {
      case 'filters':
        this.filtersToUrl(filterQueryString);
        break;
      case 'search':
        const search = this.searchQuery[filterName].replace(/\s/g, '+');
        if ('' !== search) {
          filterQueryString[slug] = search;
        }
        break;
      case 'currentPage':
        const page = this.searchQuery[filterName];
        // page 1 is default so yeah why show that?
        if (page > 1) {
          filterQueryString[slug] = this.searchQuery[filterName];
        }
        break;
      default:
        if ('' !== this.searchQuery[filterName]) {
          filterQueryString[slug] = this.searchQuery[filterName];
        }
    }
  });

  const newSearchString = new URLSearchParams(filterQueryString).toString();

  this.pageUrl.search = newSearchString;
  window.historyInitiated = true;
  window.history.pushState(null, document.title, this.pageUrl);
};

/**
 * Update the filters from the current URL
 */
lunrItemsjs.prototype.filtersToUrl = function (filters) {
  Object.keys(this.searchQuery.filters).map((filterName) => {
    let slug = this.lookups['slugs'][filterName];
    let query = this.searchQuery.filters[filterName].map((filter) => {
      return `${filter}`;
    }).join('.');
    if (query) {
      filters[slug] = query;
    }
  });
};

/**
 *  Execute this to perform a search against both Lunr and ItemsJS
 */
lunrItemsjs.prototype.search = function () {
  this.preProcessFilters();
  this.executeSearch();
  this.updateURLfromSearchQuery();
  this.updatePagination();
};

/**
 * Add the score of a boosted field to the resultsMap
 */
lunrItemsjs.prototype.lunrSearchBoostField = function (searchQuery, searchResultMap) {
  if (this.boostField) {
    const words = searchQuery.split(' ');
    let boostQuery = '';
    const boostField = this.boostField;
    words.forEach(function (word) {
      boostQuery += `+${boostField}:${word} `;
    });
    const boostResults = this.lunr.search(boostQuery);
    boostResults.forEach(result => {
      // boost search result score if
      if (searchResultMap.has(result.ref)) {
        let score = searchResultMap.get(result.ref);
        score = score + result.score;
        searchResultMap.set(result.ref, score);
      }
    });
  }
};

/**
 * Perform a search using at external Lunr.
 */
lunrItemsjs.prototype.lunrSearch = function () {
  let searchQuery = '';
  if (this.searchQuery.search.length >= this.minChars) {
    searchQuery = `${this.searchQuery.search}`;
    let searchResultMap = this.mapLunrSearchResults(this.lunr.search(searchQuery));
    this.lunrSearchBoostField(searchQuery, searchResultMap);
    return searchResultMap;
  }
  const searchResults = this.lunr.search(searchQuery);
  return this.mapLunrSearchResults(searchResults);
};

/**
 * Map the scores of a LunrSearch back.
 * @param searchResults
 * @returns {Map<any, any>}
 */

lunrItemsjs.prototype.mapLunrSearchResults = function (searchResults) {
  const searchResultMap = new Map();
  searchResults.forEach(result => {
    searchResultMap.set(result.ref, result.score);
  });
  return searchResultMap;
};
/**
 * Executes a search query on the items stored in the lunr search index.
 */
lunrItemsjs.prototype.executeSearch = function () {
  let searchResultMap = this.lunrSearch();
  let ids = [];
  searchResultMap.forEach(function (score, key) {
    ids.push(key);
  });

  if (this.preFilter && this.preFilter instanceof Function) {
    ids = this.preFilter(ids);
  }
  const itemsPerPage = ('-1' === this.itemsPerPage) ? this.documents.length : this.itemsPerPage;
  // Do not sort if there is full text query executed, because we want to boost titles first.
  const itemsjsSearchQuery = {
    ids: ids,
    page: this.searchQuery.currentPage,
    per_page: itemsPerPage,
    filters: this.searchQuery.filters,
  };
  if (!this.searchQuery.search) {
    itemsjsSearchQuery.sort = this.activeSort;
  }
  const searchResults = this.itemsjs.search(itemsjsSearchQuery);
  if (this.searchQuery.search) {
    searchResults.data.items.forEach(result => {
      result._searchRank = searchResultMap.get(result._itemsjs_id);
    });
    searchResults.data.items.sort((a, b) => b._searchRank - a._searchRank);
  }
  this.currentPage = searchResults.pagination.page;
  this.totalResults = searchResults.pagination.total;
  let listItems = searchResults.data.items.map((result) => {
    return { listItem: result[this.listItem] };
  });

  if (this.postFilter && this.postFilter instanceof Function) {
    listItems = this.postFilter(listItems);
  }
  this.searchResults = listItems;
  this.hasResults = listItems.length > 0;

  this.activeFilters = searchResults.data.aggregations;
  Object.keys(searchResults.data.aggregations).map(filterId => {
    searchResults.data.aggregations[filterId]['buckets'].map(bucket => {
      bucket['label'] = this.lookups[filterId][bucket['key']];
      return bucket;
    });
  });

  this.hasResults = this.searchResults.length > 0;
};

/**
 * Hard reset of the search
 */
lunrItemsjs.prototype.emptySearchQuery = function () {
  this.searchQuery = {
    filters: [],
    search: '',
    currentPage: 1,
  };
  Object.keys(this.filters).map(filter => {
    this.searchQuery.filters[filter] = [];
  });
};

/**
 * Build the lookup table for the filters
 */
lunrItemsjs.prototype.preProcessDocuments = function () {
  const filters = Object.keys(this.itemsjsConfiguration.aggregations);
  this.documents.forEach((doc, index) => {
    filters.forEach(filterDelta => {
      const filterBuffer = [];
      for (const [filterKey, filterVal] of Object.entries(doc[filterDelta])) {
        if (0 === filterVal.length) {
          delete this.documents[index][filterDelta];
        } else {
          if (!this.lookups.hasOwnProperty(filterDelta)) {
            this.lookups[filterDelta] = [];
            this.lookups[filterDelta][filterKey] = filterVal;
          } else if (!this.lookups[filterDelta].hasOwnProperty(filterKey)) {
            this.lookups[filterDelta][filterKey] = filterVal;
          }
          filterBuffer.push(filterKey);
        }
      }
      this.documents[index][filterDelta] = filterBuffer;
    });
  });
};
/**
 * Build the lookup table for the url slugs
 */
lunrItemsjs.prototype.initSlugLookups = function () {
  if (!this.lookups.hasOwnProperty('slugs')) {
    this.lookups.slugs = [];
  }
  this.lookups.slugs['search'] = Drupal.t('search');
  this.lookups.slugs['currentPage'] = Drupal.t('page');
  Object.keys(this.filters).forEach(filter => {
    this.lookups.slugs[this.filters[filter]['field']] = this.filters[filter]['title_slug'];
  });
};

/**
 * Pre-processes the filters before performing search on itemsjs.
 * Currently used to handle checkbox and single select filters
 */
lunrItemsjs.prototype.preProcessFilters = function () {
  Object.keys(this.searchQuery.filters).forEach(filter => {
    if (typeof this.searchQuery.filters[filter] === 'string') {
      this.searchQuery.filters[filter] = this.searchQuery.filters[filter].length > 0 ? [this.searchQuery.filters[filter]] : [];
    }
  });
};

/**
 * Load the data from the endpoint.
 */
lunrItemsjs.prototype.loadData = function () {
  if (!this.isLoading) {
    this.isLoading = true;
    const _this = this;
    const xhr = new XMLHttpRequest();
    xhr.open('GET', this.documentSourceEndoint);
    xhr.responseType = 'json';
    xhr.onload = function () {
      if (this.readyState === 4) {
        _this.documents = this.response;
        _this.initView();
      } else {
        _this.isLoading = false;
        _this.hasError = true;
      }
    };
    xhr.send();
  }
};

document.addEventListener('alpine:init', () => {
  Alpine.data('lunrItemsjsView', () => ({
        lunrItemsjs: new lunrItemsjs(drupalSettings.lunrItemsjsSettings),
        init() { this.lunrItemsjs.loadData();},
      })
  );
});



