<?php

namespace Drupal\views_lunr_itemsjs\Plugin\views\field;


use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\Core\Session\UserSession;
use Drupal\Core\Theme\ThemeInitializationInterface;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\user\RoleInterface;
use Drupal\views\Plugin\views\field\RenderedEntity;
use Drupal\views\ResultRow;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 *  Provides a field handler which renders an entity in a certain view mode
 *  in the default theme as user having a certain role
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("extended_rendered_entity")
 */
class ExtendedRenderedEntity extends RenderedEntity {


  /**
   * The theme manager.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * The account switching service.
   *
   * @var \Drupal\Core\Session\AccountSwitcherInterface
   */
  protected $accountSwitcher;

  /**
   * The theme initialization.
   *
   * @var \Drupal\Core\Theme\ThemeInitializationInterface
   */
  protected $themeInitialization;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * The entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * @inheritDoc
   */
  protected function allowAdvancedRender() {
    return FALSE;
  }
  /**
   * Constructs a new RenderedItem instance.
   *
   * @param array $configuration
   *   The plugin configuration, i.e. an array with configuration values keyed
   *   by configuration option name. The special key 'context' may be used to
   *   initialize the defined contexts by setting it to an array of context
   *   values keyed by context names.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $theme_manager
   *   The theme manager.
   * @param \Drupal\Core\Session\AccountSwitcherInterface $account_switcher
   *   The account switching service.
   * @param \Drupal\Core\Theme\ThemeInitializationInterface $theme_initialization
   *   The theme initialization.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ThemeManagerInterface $theme_manager, AccountSwitcherInterface $account_switcher, ThemeInitializationInterface $theme_initialization, ConfigFactoryInterface $config_factory, RendererInterface $renderer, EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager, EntityRepositoryInterface $entity_repository, EntityDisplayRepositoryInterface $entity_display_repository) {
    parent::__construct($configuration, $plugin_id, $plugin_definition,$entity_type_manager,$language_manager,$entity_repository,$entity_display_repository);
    $this->themeManager = $theme_manager;
    $this->accountSwitcher = $account_switcher;
    $this->themeInitialization = $theme_initialization;
    $this->configFactory = $config_factory;
    $this->renderer = $renderer;
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->entityRepository = $entity_repository;
    $this->entityDisplayRepository = $entity_display_repository;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('theme.manager'),
      $container->get('account_switcher'),
      $container->get('theme.initialization'),
      $container->get('config.factory'),
      $container->get('renderer'),
      $container->get('entity_type.manager'),
      $container->get('language_manager'),
      $container->get('entity.repository'),
      $container->get('entity_display.repository')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['role'] = [AccountInterface::ANONYMOUS_ROLE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $roles = user_role_names();
    $form['role'] = [
      '#type' => 'select',
      '#title' => $this->t('User role'),
      '#description' => $this->t('Your item will be rendered as seen by a user with the selected role. We recommend to just use "@anonymous" here to prevent data leaking out to unauthorized roles.', ['@anonymous' => $roles[AccountInterface::ANONYMOUS_ROLE]]),
      '#options' => $roles,
      '#multiple' => FALSE,
      '#default_value' => $this->options['role'],
      '#required' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $entity = $this->getEntity($values);
    if ($entity === NULL) {
      return '';
    }
    $entity = $this->getEntityTranslationByRelationship($entity, $values);
    return $this->renderAsUserInViewMode($entity);
  }

  public function renderAsUserInViewMode($entity): array {
    $build = [];
    $active_theme_switched = FALSE;
    $active_theme = $this->themeManager->getActiveTheme();
    try {
      // Switch to desired Theme if needed
      $default_theme = $this->configFactory->get('system.theme')
        ->get('default');
      $default_theme = $this->themeInitialization->getActiveThemeByName($default_theme);

      if ($default_theme->getName() !== $active_theme->getName()) {
        $this->themeManager->setActiveTheme($default_theme);
        // Ensure that static cached default variables are set correctly,
        // especially the directory variable.
        drupal_static_reset('template_preprocess');
        $active_theme_switched = TRUE;
      }

      /**
       * If a (non-anonymous) role is selected,
       * then also add the authenticated user role.
       */
      $roles = [$this->options['role']];
      $authenticated = RoleInterface::AUTHENTICATED_ID;
      if (array_diff($roles, [$authenticated, RoleInterface::ANONYMOUS_ID])) {
        $roles[$authenticated] = $authenticated;
      }
      $this->accountSwitcher->switchTo(new UserSession(['roles' => array_values($roles)]));
      $access = $entity->access('view', NULL, TRUE);
      $build['#access'] = $access;
      if ($access->isAllowed()) {
        $view_builder = $this->entityTypeManager->getViewBuilder($this->getEntityTypeId());
        $builder = [];
        $builder['#access'] = $access;
        $builder += $view_builder->view($entity, $this->options['view_mode'], $entity->language()->getId());
        $value = (string) $this->renderer->render($builder);
        $build['#markup'] =$value;
      }
    } catch (\Exception $e) {
      \Drupal::logger('views_lunr_itemsjs')->error($e->getMessage());
    }
    $this->accountSwitcher->switchBack();
    if ($active_theme_switched) {
      $this->themeManager->setActiveTheme($active_theme);
      // Ensure that static cached default variables are set correctly,
      // especially the directory variable.
      drupal_static_reset('template_preprocess');
    }
    return $build;

  }

}
