<?php

namespace Drupal\views_lunr_itemsjs\Form;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Views;
use Drupal\views_lunr_itemsjs\Plugin\views\row\LunrSearchIndexRow;

class LunrItemsjsConfigurationForm extends EntityForm {

  /**
   * @var \Drupal\views_lunr_itemsjs\Entity\lunrItemsjs
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#description' => $this->t('Label for the search configuration.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => '\Drupal\views_lunr_itemsjs\Entity\lunrItemsjs::load',
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $this->entity->get('description'),
      '#description' => $this->t('Description of the search configuration.'),
    ];

    $form['view'] = [
      '#type' => 'select',
      '#title' => $this->t('Views source'),
      '#options' => $this->getLunrViews(),
      '#default_value' => $this->entity->getViewID(),
      '#required' => TRUE,
    ];

    $form['display'] = [
      '#type' => 'select',
      '#title' => $this->t('Display type'),
      '#options' => [
        'overview' => $this->t('Overview of items with filters.'),
        'search' => $this->t('Autocomplete search.'),
      ],
      '#default_value' => $this->entity->getDisplayType(),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateDisplayWrapper',
        'wrapper' => 'display-wrapper',
      ],

      '#states' => [
        'invisible' => [
          [':input[name="index"]' => ['value' => '']],
        ],
      ],
    ];

    $form['display_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'display-wrapper'],
    ];

    $view = $this->entity->getViewID() ?? $form_state->getValue('view');
    $display = $form_state->getValue('display') ?? $this->entity->getDisplayType();

    if (!empty($view)) {
      $availableFields = $this->getIndexFields($view);
      $settings = $this->entity->getSettings();
      $form['display_wrapper']['settings'] = [
        '#type' => 'details',
        '#collapsible' => TRUE,
        '#collapsed' => FALSE,
        '#open' => TRUE,
        '#title' => $this->t('Settings'),
        '#tree' => TRUE,
      ];

      $form['display_wrapper']['settings']['multilang'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Content contains dutch words.'),
        '#default_value' => $settings['multilang'],
      ];

      $form['display_wrapper']['settings']['list_item'] = [
        '#type' => 'select',
        '#required' => TRUE,
        '#empty_option:' => $this->t('-Select-'),
        '#title' => $this->t('List item'),
        '#multiple' => FALSE,
        '#options' => $availableFields,
        '#default_value' => $this->entity->getListItem(),
        '#description' => $this->t('This field will used to display the results.'),
        '#parents' => ['list_item'],
      ];

      $form['display_wrapper']['settings']['indexed_fields'] = [
        '#type' => 'checkboxes',
        '#required' => TRUE,
        '#empty_option:' => $this->t('-Select-'),
        '#title' => $this->t('Searchable fields.'),
        '#multiple' => TRUE,
        '#options' => $availableFields,
        '#default_value' => $this->entity->getIndexedFields(),
        '#description' => $this->t('These Fields are use to build the Lunr full text index.'),
        '#parents' => ['indexed_fields'],
        '#ajax' => [
          'callback' => '::updateBoostedFields',
          'wrapper' => 'boosted-fields-wrapper',
        ],
      ];

      $boostedFieldOptions = array_intersect_key($availableFields, array_flip($this->entity->getIndexedFields()));

      $form['display_wrapper']['settings']['boosted_field'] = [
        '#type' => 'select',
        '#empty_option:' => $this->t('-Select-'),
        '#title' => $this->t('Boostable field'),
        '#multiple' => FALSE,
        '#options' => $boostedFieldOptions,
        '#empty_value' => FALSE,
        '#default_value' => $settings['boosted_field'],
        '#description' => $this->t('Boost this searchable field in the results.'),
        '#prefix' => '<div id="boosted-fields-wrapper">',
        '#suffix' => '</div>',
      ];

      $form['display_wrapper']['settings']['min_chars'] = [
        '#type' => 'number',
        '#title' => $this->t('Minimal length to execute a full text search.'),
        '#min' => 3,
        '#max' => 8,
        '#default_value' => $settings['min_chars'] ?? 3,
        '#description' => $this->t('The minimum length for a search to be performed, range is between 3 and 8 characters.'),
      ];

      if ($display === 'search') {
        $this->searchSettingsForm($form, $settings, $availableFields);
      }
      if ($display === 'overview') {
        $this->overviewSettingsForm($form, $settings, $availableFields);
      }
    }

    //    $this->formAddTranslatableStrings($form);

    return $form;
  }

  public function updateDisplayWrapper(array $form, FormStateInterface $form_state) {
    return $form['display_wrapper'];
  }

  public function updateBoostedFields($form, FormStateInterface $form_state) {
    $boosted_options = [];
    foreach ($form_state->getValue('indexed_fields') as $key => $value) {
      if ($value !== 0) {
        $boosted_options[$key] = $form['display_wrapper']['settings']['indexed_fields']['#options'][$key];
      }
    }
    $form['display_wrapper']['settings']['boosted_field']['#options'] = $boosted_options;
    $form_state->setRebuild(TRUE);
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#boosted-fields-wrapper', ($form['display_wrapper']['settings']['boosted_field'])));
    return $response;
  }

  /**
   * Form fields only needed for autocomplete view.
   *
   * @param array $form
   * @param array $settings
   * @param array $availableFields
   *
   * @return void
   */
  public function searchSettingsForm(array &$form, array $settings, array $availableFields) {
    $form['display_wrapper']['settings']['search_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Search mode'),
      '#empty_value' => '',
      '#empty_option:' => $this->t('-Select-'),
      '#multiple' => FALSE,
      '#options' => [
        'partial' => $this->t('String contains word ("*word*")'),
        'full' => $this->t('Full word match ("word")'),
        'starts' => $this->t('String starts with word ("word*)'),
        'ends' => $this->t('String ends with word ("*word)'),
      ],
      '#default_value' => $settings['search_mode'] ?? 'partial',
    ];

    $form['display_wrapper']['settings']['per_page'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Visible links'),
      '#maxlength' => 3,
      '#size' => 4,
      '#default_value' => $this->entity->getItemsPerPage(),
      '#description' => $this->t('set to -1 to show all results'),
      '#parents' => ['per_page'],
    ];

    $availableNodeTypes = array_keys($this->entityTypeManager->getStorage('node_type')
      ->loadMultiple());
    $default = $settings['target_page'];
    if ($default) {
      $default = $this->entityTypeManager->getStorage('node')->load($default);
    }
    $form['display_wrapper']['settings']['target_page'] = [
      '#type' => 'entity_autocomplete',
      '#target_type' => 'node',
      '#selection_settings' => [
        'target_bundles' => $availableNodeTypes,
      ],
      '#title' => $this->t('Destination search page'),
      '#default_value' => $default,
    ];
  }

  /**
   * Form fields only needed for overviewForm view.
   *
   * @param array $form
   * @param array $settings
   * @param array $availableFields
   *
   * @return void
   */
  public function overviewSettingsForm(array &$form, array $settings, array $availableFields) {
    $form['display_wrapper']['settings']['visible_links'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Pagination items'),
      '#maxlength' => 3,
      '#size' => 4,
      '#default_value' => $this->entity->getVisibleLinks(),
      '#description' => $this->t('The amount of visible pagination links'),
      '#parents' => ['visible_links'],
    ];

    $form['display_wrapper']['settings']['per_page'] = [
      '#type' => 'textfield',
      '#required' => TRUE,
      '#title' => $this->t('Results per page'),
      '#maxlength' => 3,
      '#size' => 4,
      '#default_value' => $this->entity->getItemsPerPage(),
      '#description' => $this->t('set to -1 to show all results'),
      '#parents' => ['per_page'],
    ];

    $form['display_wrapper']['settings']['cache'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Browser cache behaviour.'),
    ];

    $form['display_wrapper']['settings']['cache']['cache_max_age'] = [
      '#type' => 'number',
      '#required' => TRUE,
      '#title' => $this->t('Cache time in seconds'),
      '#description' => $this->t('Cache of the data in the browser (Cache-Control: max-age), set to 0 to disable cache. Max allowed is 86400 seconds (1 day)'),
      '#min' => 0,
      '#max' => 86400,
      '#default_value' => $settings['cache_max_age'] ?? 900,
      '#parents' => ['settings', 'cache_max_age'],
    ];
    $form['display_wrapper']['settings']['cache']['cache_public'] = [
      '#type' => 'checkbox',
      '#required' => TRUE,
      '#title' => $this->t('Shared cache'),
      '#description' => $this->t('Uncheck if this cache is generated per user (Cache-Control: private)'),
      '#min' => 0,
      '#max' => 86400,
      '#default_value' => $settings['cache_public'] ?? 1,
      '#parents' => ['settings', 'cache_public'],
    ];

    $form['display_wrapper']['display_wrapper']['filters'] = [
      '#parents' => ['filters'],
      '#type' => 'element_multiple',
      '#title' => $this->t('Filters'),
      '#item_label' => $this->t('filter'),
      '#add_more' => TRUE,
      '#add_more_items' => 1,
      '#min_items' => 1,
      '#add_more_button_label' => $this->t('Add'),
      '#add_more_input' => TRUE,
      '#add_more_input_label' => $this->t('Filter(s)'),
      '#header' => [
        ['data' => $this->t('Label'), 'width' => '20%'],
        ['data' => $this->t('Source'), 'width' => '20%'],
        ['data' => $this->t('Behaviour'), 'width' => '12%'],
        ['data' => $this->t('Limit'), 'width' => '12%'],
        ['data' => $this->t('Sort on'), 'width' => '8%'],
        ['data' => $this->t('Order by'), 'width' => '8%'],
        ['data' => $this->t('Selected 1st'), 'width' => '10%'],
        ['data' => $this->t('Hide empty'), 'width' => '10%'],
      ],
      '#default_value' => $this->entity->getFilters(),
      '#add_empty' => TRUE,
      '#element' => [
        'label' => [
          '#type' => 'textfield',
          '#title' => $this->t('Filter label'),
          '#required' => TRUE,
        ],
        'field' => [
          '#type' => 'select',
          '#empty_value' => '',
          '#empty_option:' => $this->t('-Select-'),
          '#title' => $this->t('Source field'),
          '#multiple' => FALSE,
          '#options' => $availableFields,
        ],
        'conjunction' => [
          '#type' => 'radios',
          '#title' => $this->t('Facet behaviour (Conjunction Type)'),
          '#options' => [
            '1' => $this->t('Narrow results (AND)'),
            '0' => $this->t('Expand results (OR)'),
          ],
          '#default_value' => 1,
        ],
        'size' => [
          '#type' => 'number',
          '#title' => $this->t('Max results'),
          '#min' => 1,
          '#max' => 50,
          '#step' => 1,
          '#default_value' => 20,
        ],
        'sort' => [
          '#type' => 'radios',
          '#required' => TRUE,
          '#title' => $this->t('Sort by'),
          '#default_value' => 'count',
          '#options' => [
            'count' => $this->t('Count'),
            'key' => $this->t('Label'),
          ],
        ],
        'order' => [
          '#type' => 'radios',
          '#title' => $this->t('Sort order'),
          '#default_value' => 'asc',
          '#options' => [
            'asc' => $this->t('Ascending'),
            'desc' => $this->t('Descending'),
          ],
        ],
        'selected_first' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Show selected items first in the list'),
          '#default_value' => 1,
        ],
        'hide_zero' => [
          '#type' => 'checkbox',
          '#title' => $this->t('Hide facets with no results'),
          '#default_value' => 1,
        ],

      ],
    ];

    $form['display_wrapper']['sorts'] = [
      '#parents' => ['sorts'],
      '#type' => 'element_multiple',
      '#title' => $this->t('Result sort orders'),
      '#item_label' => $this->t('filter'),
      '#add_more' => TRUE,
      '#min_items' => 1,
      '#add_more_button_label' => $this->t('Add Sort'),
      '#add_more_input' => FALSE,
      '#header' => [
        ['data' => $this->t('Label'), 'width' => '30%'],
        ['data' => $this->t('Configuration'), 'width' => '60%'],
        ['data' => $this->t('Default'), 'width' => '10%'],
      ],
      '#default_value' => $this->entity->get('sorts'),
      '#add_empty' => FALSE,
      '#element' => [
        'label' => [
          '#type' => 'textfield',
          '#title' => $this->t('Sort label'),
          '#required' => TRUE,
        ],
        'config' => [
          '#title' => $this->t('Configuration'),
          '#required' => TRUE,
          '#type' => 'textfield',
        ],

        'default_sort' => [
          '#type' => 'radios',
          '#title' => $this->t('Default Sort'),
          '#options' => [
            '1' => $this->t('Yes'),
            '0' => $this->t('No'),
          ],
        ],
      ],
    ];

    $fields = "_searchRank, " . implode(", ", array_keys($availableFields));
    $form['display_wrapper']['sorts']['available_fields'] = [
      '#weight' => 100,
      '#attributes' => ['class' => 'form-item__description'],
      '#theme_wrappers' => [
        'form_element',
      ],
      '#type' => 'item',
      '#markup' => $this->t('The format for sorts is <pre>fieldname:order, otherfield:order</pre>Order can be asc or desc<br><br><i>_searchRank</i> is an internal field that contains the relative score of a record after a free text search.<hr/>Available fields for sorts <pre>%fields</pre>', ['%fields' => $fields]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $message = $result === SAVED_NEW
      ? $this->t('Created Lunr-ItemsJS configuration %label.', $message_args)
      : $this->t('Updated Lunr-ItemsJS configuration %label.', $message_args);
    $this->messenger()->addStatus($message);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

  public function getLunrViews(): array {
    $displays = Views::getApplicableViews('lunr_search_index');
    $options = [];
    foreach ($displays as $display) {
      [$view_id, $display_id] = $display;
      $view = $this->entityTypeManager->getStorage('view')->load($view_id);
      $options[$view_id . '.' . $display_id] = $this->t('@view : @display', [
        '@view' => $view->label(),
        '@display' => $view->get('display')[$display_id]['display_title'],
      ]);
    }
    return $options;
  }

  /**
   * Get the avaiable fields for the chosen view
   *
   * @param string $index
   *
   * @return array|null
   */
  public function getIndexFields(string $index): ?array {
    if (!$index) {
      return [];
    }

    if (!$this->getEntity()->get('view')) {
      $this->getEntity()->set('view', $index);
    }
    $view = $this->getEntity()->getViewExecutable();
    $fields = [];
    if ($view) {
      $fieldOptions = $view->display_handler->getOption('row')['options']['field_options'];
      $aliases = LunrSearchIndexRow::extractFromOptionsArray('alias', $fieldOptions);
      foreach ($view->display_handler->getOption('fields') as $field_name => $field_data) {
        $alias = $aliases[$field_name];
        if (!$alias) {
          $alias = $field_name;
        }
        $label = $field_data['admin_label'];
        if (!$label) {
          $label = $field_data['id'];
        }
        $fields[$alias] = $label;
      }
    }
    return $fields;
  }

}
