<?php

namespace Drupal\views_lunr_itemsjs\Plugin\Block;


use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Url;
use Drupal\node\Entity\Node;
use Drupal\views_lunr_itemsjs\lunrItemsConfigInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;


/**
 *
 * @Block(
 *   id = "lunr_itemsjs_block",
 *   admin_label = @Translation("Lunr-ItemsJS Block"),
 *   category = @Translation("Views")
 * )
 */
class LunrItemsjsBlock extends BlockBase implements ContainerFactoryPluginInterface {


  protected EntityTypeManagerInterface $entityTypeManager;


  protected ?lunrItemsConfigInterface $lunrItemsConfig;

  protected LanguageManagerInterface $languageManager;

  private \StdClass $buildSettings;


  protected array $blockBuild = [];

  public function __construct(array                      $configuration, $plugin_id, $plugin_definition,
                              EntityTypeManagerInterface $entity_type_manager,
                              LanguageManagerInterface   $languageManager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $languageManager;
    $this->buildSettings = new \stdClass();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('language_manager')
    );
  }


  public function defaultConfiguration(): array {
    $configuration['search_configuration'] = NULL;
    return $configuration;
  }


  public function blockSubmit($form, FormStateInterface $form_state) {
    $this->configuration['search_configuration'] = $form_state->getValue('search_configuration');
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {

    $this->initLunrItemsJSConfig();
    $this->initBuildSettings();
    if ($this->lunrItemsConfig->getDisplayType() === 'overview') {
      $this->buildOverviewConfigation();
    }
    else {
      $this->buildAutocompleteConfigation();
    }
    $this->blockBuild['content']['#view'] = $this->configuration['search_configuration'];
    // no cache... at this moment. Might look into cache context later.
    $this->blockBuild['#cache'] = [
      'max-age' => 0,
    ];
    $build = $this->blockBuild;
    $build['#attached']['drupalSettings'] = [
      'lunrItemsjsSettings' => $this->buildSettings,
    ];
    \Drupal::moduleHandler()
      ->invokeAll('lunr_itemsjs_configuration_alter', [&$build]);
    return $build;
  }

  /**
   * Load the LunrItemsJS configuration
   */
  protected function initLunrItemsJSConfig() {
    $entityStorage = $this->entityTypeManager->getStorage('lunr_itemsjs');
    $this->lunrItemsConfig = $entityStorage->load($this->configuration['search_configuration']);
    if (!$this->lunrItemsConfig) {
      throw new EntityStorageException($this->t('Lunr-ItemsJS config "%config_id" was not found', ['%config_id' => $this->configuration['search_configuration']]));
    }
  }

  /**
   * Add the configuration for the Overview block to the build array.
   */
  protected function buildOverviewConfigation(): void {
    $this->buildSettings->filters = $this->lunrItemsConfig->buildBlockFilters();
    $this->buildSettings->sorts = $this->lunrItemsConfig->buildBlockSorts();
    $this->blockBuild['content']['#theme'] = 'lunritemsjs_view';
    $this->blockBuild['#attached']['library'] = 'views_lunr_itemsjs/drupal.itemsjs';
  }

  /**
   * Add the configuration for the Autocomplete block to the build array.
   */
  protected function buildAutocompleteConfigation(): void {
    $this->blockBuild['content']['#theme'] = 'lunritemsjs_autocomplete';
    $this->blockBuild['#attached']['library'] = 'views_lunr_itemsjs/drupal.autocomplete';
    $target = FALSE;
    if ($this->lunrItemsConfig->getSetting('target_page')) {
      $language = $this->languageManager->getCurrentLanguage(LanguageInterface::TYPE_CONTENT);
      $node = Node::load($this->lunrItemsConfig->getSetting('target_page'));
      if ($node){
        $target = $this->buildSettings->target_page =  $node->toUrl('canonical',['language'=> $language])->toString(false);
      }
    }
    $this->buildSettings->target_page  = $target;
  }


  public function initBuildSettings(): void {
    $language = $this->languageManager->getCurrentLanguage();
    $sourceURL = URL::fromRoute('views_lunr_itemsjs.index',
      ['lunr_itemsjs' => $this->lunrItemsConfig->id()],
      ['language' => $language])->toString(FALSE);

    $this->buildSettings->source = $sourceURL;
    $this->buildSettings->list_item = $this->lunrItemsConfig->getListItem();
    $this->buildSettings->language = $language->getId();
    $this->buildSettings->per_page = $this->lunrItemsConfig->getItemsPerPage();
    $this->buildSettings->pagination_visble_links = $this->lunrItemsConfig->getVisibleLinks();
    $this->buildSettings->indexed_fields = $this->lunrItemsConfig->getIndexedFields();
    $this->buildSettings->settings = $this->lunrItemsConfig->getSettings();
  }

  public function buildConfigurationForm($form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $availableConfigs = $this->entityTypeManager->getListBuilder('lunr_itemsjs')
      ->load();
    if (!$availableConfigs) {
      $form['error'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Error'),
        '#markup' => $this->t('<strong class="error">Create a <a href=":config">Lunr ItemsJS configuration</a> first, before adding a block.</strong>',
          [
            ':config' => Url::fromRoute('entity.lunr_itemsjs.add_form')
              ->toString(),
          ]),
      ];
      return $form;
    }

    $options = [];
    foreach ($availableConfigs as $config) {
      $options[$config->id()] = $config->label();
    }

    $form['search_configuration'] = [
      '#type' => 'select',
      '#required' => TRUE,
      '#empty_value' => '',
      '#empty_option:' => $this->t('-Select-'),
      '#title' => $this->t('Search configuration'),
      '#multiple' => FALSE,
      '#options' => $options,
      '#default_value' => $this->configuration['search_configuration'],
    ];
    return $form;
  }

}
