<?php

namespace Drupal\views_lunr_itemsjs\Plugin\views\style;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * The style plugin for serialized output formats.
 *
 * This class is largely based on the core REST module.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "lunr_search_index_json",
 *   title = @Translation("LunrItemsjs search index JSON"),
 *   help = @Translation("Serializes views row data to JSON."),
 *   display_types = {"lunr_search_index"}
 * )
 */
class LunrSearchIndexJson extends StylePluginBase implements CacheableDependencyInterface {

  /**
   * {@inheritdoc}
   */
  protected $usesRowPlugin = TRUE;

  /**
   * {@inheritdoc}
   */
  protected $usesGrouping = FALSE;

  /**
   * The serializer which serializes the views result.
   *
   * @var \Symfony\Component\Serializer\Serializer
   */
  protected $serializer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('serializer')
    );
  }

  /**
   * Constructs a Plugin object.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, SerializerInterface $serializer) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->serializer = $serializer;
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $rows = [];
    $filters = [];
    $fieldOptions = $this->view->getDisplay()->getOption('row')['options']['field_options'];
    foreach ($fieldOptions as $fieldName => $fieldOption){
      if ( (bool) $fieldOption['filter']){
        $filters[] = $fieldOption['alias'] ?: $fieldName;
      }
    }

    foreach ($this->view->result as $row_index => $row) {
     $this->view->row_index = $row_index;
      // To speed up the context switch rendering, grouping AllExtend entities in 1 go might work!
      $output = $this->view->rowPlugin->render($row);
      $_itemsjs_id = $output['_itemsjs_id'];
      if (!$rows[$_itemsjs_id]){
        $rows[$_itemsjs_id] = $output;
      } else {
        foreach ($filters as $filter){
          $filterValue = $output[$filter];
          $rows[$_itemsjs_id][$filter] += $filterValue;
        }
      }
    }
    unset($this->view->row_index);
    return $this->serializer->serialize(array_values($rows), 'json');
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return Cache::PERMANENT;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = parent::calculateDependencies();
    $dependencies['module'][] = 'serialization';
    return $dependencies;
  }

}
