<?php

namespace Drupal\views_lunr_itemsjs\Plugin\views\row;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\row\RowPluginBase;
use Drupal\views_lunr_itemsjs\Entity\lunrItemsjs;
use Drupal\views_lunr_itemsjs\Plugin\views\ExtractFromOptionsTrait;
use Drupal\views_lunr_itemsjs\Plugin\views\field\ExtendedRenderedEntity;

/**
 * Plugin which displays fields as raw data.
 *
 * This class is largely based on the core REST module.
 *
 * @ingroup views_row_plugins
 *
 * @ViewsRow(
 *   id = "lunr_search_index_row",
 *   title = @Translation("LunrItemsjs search index row"),
 *   help = @Translation("Use fields in index."),
 *   display_types = {"lunr_search_index"}
 * )
 */
class LunrSearchIndexRow extends RowPluginBase {

  use ExtractFromOptionsTrait;
  /**
   * {@inheritdoc}
   */
  protected $usesFields = TRUE;

  /**
   * Stores an array of prepared field aliases from options.
   *
   * @var array
   */
  protected array $replacementAliases = [];

  /**
   * Stores an array of options to determine if the raw field output is used.
   *
   * @var array
   */
  protected array $rawOutputFields = [];

  /**
   * Stores an array of options to determine if the field is a filter.
   *
   * @var array
   */
  protected array $filterOutputFields = [];

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL): void {
    parent::init($view, $display, $options);

    if (!empty($this->options['field_options'])) {
      $options = (array) $this->options['field_options'];
      // Prepare a trimmed version of replacement aliases.
      $aliases = static::extractFromOptionsArray('alias', $options);
      $this->replacementAliases = array_filter(array_map('trim', $aliases));
      // Prepare an array of raw output field options.
      $this->rawOutputFields = static::extractFromOptionsArray('raw_output', $options);
      $this->filterOutputFields = static::extractFromOptionsArray('filter', $options);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['field_options'] = ['default' => []];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    $form['field_options'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Field'),
        $this->t('Alias'),
        $this->t('Raw output'),
        $this->t('Filter'),
      ],
      '#empty' => $this->t('You have no fields. Add some to your view.'),
      '#tree' => TRUE,
    ];

    $options = $this->options['field_options'];

    if ($fields = $this->view->display_handler->getOption('fields')) {
      foreach ($fields as $id => $field) {
        // Don't show the field if it has been excluded.
        if (!empty($field['exclude'])) {
          continue;
        }
        $form['field_options'][$id]['field'] = [
          '#markup' => ($field['admin_label'] !== '') ? "$id (${field['admin_label']})" : $id,
        ];
        $form['field_options'][$id]['alias'] = [
          '#title' => $this->t('Alias for @id', ['@id' => $id]),
          '#title_display' => 'invisible',
          '#type' => 'textfield',
          '#default_value' => $options[$id]['alias'] ?? '',
          '#element_validate' => [[$this, 'validateAliasName']],
        ];
        $form['field_options'][$id]['raw_output'] = [
          '#title' => $this->t('Raw output for @id', ['@id' => $id]),
          '#title_display' => 'invisible',
          '#type' => 'checkbox',
          '#default_value' => $options[$id]['raw_output'] ?? '',
        ];
        $form['field_options'][$id]['filter'] = [
          '#title' => $this->t('@id is a filter', ['@id' => $id]),
          '#title_display' => 'invisible',
          '#type' => 'checkbox',
          '#default_value' => $options[$id]['filter'] ?? '',
        ];
      }
    }
  }

  /**
   * Form element validation handler.
   */
  public function validateAliasName($element, FormStateInterface $form_state): void {
    if (preg_match('@[^A-Za-z0-9_-]+@', $element['#value'])) {
      $form_state->setError($element, $this->t('The machine-readable name must contain only letters, numbers, dashes and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state): void {
    // Collect an array of aliases to validate.
    $aliases = static::extractFromOptionsArray('alias', $form_state->getValue([
      'row_options',
      'field_options',
    ]));

    // If array filter returns empty, no values have been entered. Unique keys
    // should only be validated if we have some.
    if (($filtered = array_filter($aliases)) && (array_unique($filtered) !== $filtered)) {
      $form_state->setErrorByName('aliases', $this->t('All field aliases must be unique'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render($row): array {
    $output = [];

    foreach ($this->view->field as $id => $field) {
      // Omit excluded fields from the rendered output.
      if (empty($field->options['exclude'])) {
        // If the raw output option has been set, just get the raw value.
        if (!empty($this->rawOutputFields[$id])) {
          $value = $field->getValue($row);
        }
        else if ($field instanceof ExtendedRenderedEntity) {
          /**
           * The extend field renders in different user context
           * and advancedRender does to much processing afterwards.
           */
          $value = $field->render($row)['#markup'];
        }
        else {
          $value = $field->advancedRender($row);
        }
        /**
         * Filters can be based on raw or rendered output
         * that's why they are handled separately
         */
        if (!empty($this->filterOutputFields[$id])) {
          $safeName = lunrItemsjs::machineName($value);
          $value = [$safeName => (string) $value];
        }
        // handle empty values? or handle them on indexer level?
        $output[$this->getFieldKeyAlias($id)] = $value;

      }
    }

    // Use a UUID as PK rather then the index counter
    $output['_itemsjs_id'] = $row->_entity->uuid();

    return $output;
  }

  /**
   * Return an alias for a field ID, as set in the options form.
   *
   * @param string $id
   *   The field id to lookup an alias for.
   *
   * @return string
   *   The matches user entered alias, or the original ID if nothing is found.
   */
  public function getFieldKeyAlias(string $id): string {
    return $this->replacementAliases[$id] ?? $id;
  }


}
