function lunrItemsjs(settings) {
  this.documentSourceEndoint = settings.source;
  this.pageUrl = new URL(window.location.href);

  this.language = settings.language;
  this.strings = {...settings.strings || {}};
  this.emptyResultString = 'No results found.'; // Not sure to keep this


  // The Facet filters
  this.filters = settings.filters;
  // Lunr Searchable fields
  this.indexedFields = settings.indexed_fields;
  // Available sort options
  this.sorts = settings.sorts;
  // Primary key of indexed items, hardcoded in the view.
  this.indexPrimaryKey = settings.index_primary ?? '_itemsjs_id';

  // Item used for rendering results can be string or template
  this.listItem = settings.list_item;


  /**
   * Lookup table for internal usage.
   * @type {*[]}
   */
  this.lookups = [];

  this.activeFilters = [];
  this.totalResults = -1;


  /**
   * Loading state
   * @type {boolean}
   */
  this.isLoading = true;
  /**
   * Has load error?
   * @type {boolean}
   */
  this.hasError = false;

  this.itemsPerPage = settings?.per_page ?? -1;

  /**
   *  Indexable documents, loaded from the the documentSourceEndoint
   * @type {*[]}
   */
  this.documents = [];
  /**
   * Results for this page after a search has been performed.
   * @type { {} }
   */
  this.searchResults = {};

  /**
   * State of the current 'search'
   * @type {{filters: *[], search: string, currentPage: number}}
   */
  this.searchQuery = {
    filters: [],
    search: '',
    currentPage: 1,
  };

  this.pageUrl = new URL(window.location.href);

  this.itemsjsConfiguration = {
    sortings: this.sorts,
    aggregations: this.filters,
    searchableFields: Object.keys(this.indexedFields),
    native_search_enabled: false,
    custom_id_field: this.indexPrimaryKey
  };
  // Helper structure to make pagination easier.
  this.pagination = {
    total: this.totalResults,
    per_page: this.itemsPerPage,
    current_page: this.currentPage,
    last_page: Math.ceil(this.totalResults / this.itemsPerPage),
  };

  /**
   * Configuration for ItemsJS is rather static
   * @type {{searchableFields: string[], sortings, native_search_enabled:
   *   boolean, aggregations, custom_id_field: (*|string)}}
   */

  /**
   * Bind for fully indexed Lunr
   * @type {lunr|{}}
   */
  this.lunr = {};
  /**
   * Bind for fully indexed itemsjs
   * @type {itemsjs|{}}
   */
  this.itemsjs = {}; // Will hold the ItemsJS system

};

lunrItemsjs.prototype.initItemsJS = function () {

  this.itemsjs = itemsjs(this.documents, this.itemsjsConfiguration);
};


lunrItemsjs.prototype.initLunrSearch = function () {
  const _this = this;
  this.lunr = new lunr(function () {
    /**
     * The primary key for this index is `_itemsjs_id` as defined in the view
     * Make sure if you change that in a view, you change that in the
     * drupalSettings as well!
     *
     * Since users can add a custom field called ID it's set to this less
     * common guessable primary key by default.
     *
     * @see \Drupal\views_lunr_itemsjs\Plugin\search_api\backend\LunrBackend::indexItems
     */
    this.ref(_this.indexPrimaryKey);
    Object.keys(_this.indexedFields).map(field => {
      this.field(field);
    });
    _this.documents.forEach(function (document) {
      this.add(document);
    }, this);
  });
};

lunrItemsjs.prototype.initView = function () {

  this.preProcessDocuments();
  this.initLunrSearch();
  this.initItemsJS();
  this.initSlugLookups();
  this.emptySearchQuery();
  this.executeSearch();
  this.updateSearchQueryFromUrl()
  // If the URL has a searchString bind that to the searchQuery and redo search
  this.isLoading = false;


};

/**
 * Returns a single active filter if available
 * @param filterName
 * @returns {*|boolean}
 */
lunrItemsjs.prototype.getActiveFilter = function (filterName) {
  return (this.activeFilters.hasOwnProperty(filterName)) ? this.activeFilters[filterName] : false;
};


lunrItemsjs.prototype.updatePagination = function () {
  this.pagination = {
    total: this.totalResults,
    per_page: this.itemsPerPage,
    current_page: this.currentPage,
    last_page: Math.ceil(this.totalResults / this.itemsPerPage),
  };
}

/**
 * Update this.searchQuery fom pageUrl.searchParams
 */
lunrItemsjs.prototype.updateSearchQueryFromUrl = function () {
  let inversedLookup = [];
  for (const [key, value] of Object.entries(this.lookups['slugs'])) {
    inversedLookup[value] = key;
  }
  const searchParams = Array.from(this.pageUrl.searchParams);
  if (searchParams.length > 0) {
    searchParams.forEach(filterPairs => {
      const key = filterPairs[0]
      const filter = inversedLookup[key];
      if (this.searchQuery.filters[filter]) {
        this.searchQuery.filters[filter] = filterPairs[1].split('.');
      }
      else {
        this.searchQuery[filter] = filterPairs[1];
      }
    });
    this.executeSearch();
  }
};
/**
 * Update the page url from this.searchQuery content
 */
lunrItemsjs.prototype.updateURLfromSearchQuery = function () {
  const filterQueryString = {};
  Object.keys(this.searchQuery).map(filterName => {
    const slug = this.lookups['slugs'][filterName]
    if (filterName === 'filters') {
      this.filtersToUrl(filterQueryString)
    }
    else if (filterName === 'search') {
      const search = this.searchQuery[filterName].replace(/\s/g, '+');
      if ("" !== search) {
        filterQueryString[slug] = search;
      }
    }else if (filterName === 'currentPage'){
      const page = this.searchQuery[filterName]
      // page 1 is default so yeah why show that?
      if (page > 1){
        filterQueryString[slug] = this.searchQuery[filterName];
      }
    }
    else if ("" !== this.searchQuery[filterName]) {
      filterQueryString[slug] = this.searchQuery[filterName];
    }
  });

  const newSearchString = new URLSearchParams(filterQueryString).toString();

  this.pageUrl.search = newSearchString;
  window.historyInitiated = true;
  window.history.pushState(null, document.title, this.pageUrl);
};

lunrItemsjs.prototype.filtersToUrl = function (filters) {
  Object.keys(this.searchQuery.filters).map((filterName) => {
    let slug = this.lookups['slugs'][filterName];
    let query = this.searchQuery.filters[filterName].map((filter) => {
      return `${filter}`;
    }).join('.');
    if (query) {
      filters[slug] = query;
    }
  })
};

lunrItemsjs.prototype.search = function () {
  this.executeSearch();
  this.updatePagination();
  this.updateURLfromSearchQuery();
};

lunrItemsjs.prototype.executeSearch = function () {
  const search_result = this.lunr.search(this.searchQuery.search);
  let ids = search_result.map(v => v.ref);

  if (this.preFilter && this.preFilter instanceof Function) {
    ids = this.preFilter(ids);
  }

  const itemsPerPage = (this.itemsPerPage === -1) ? this.documents.length : this.itemsPerPage;
  const searchQuery = {
    ids: ids,
    page: this.searchQuery.currentPage,
    per_page: itemsPerPage,
    sort: 'start_date',
    filters: this.searchQuery.filters,
  };

  const results = this.itemsjs.search(searchQuery);

  this.currentPage = results.pagination.page;
  this.totalResults = results.pagination.total;
  this.searchResults = results.data.items.map((result) => {
    return {listItem: result[this.listItem]};
  });
  this.activeFilters = results.data.aggregations;
  Object.keys(results.data.aggregations).map(filterId => {
    results.data.aggregations[filterId]['buckets'].map(bucket => {
      bucket['label'] = this.lookups[filterId][bucket['key']];
      return bucket;
    })
  })
};
lunrItemsjs.prototype.emptySearchQuery = function () {
  this.searchQuery = {
    filters: [],
    search: '',
    currentPage: 1,
  };
  Object.keys(this.filters).map(filter => {
    this.searchQuery.filters[filter] = [];
  });
};

/**
 * The click target for a reset button.
 */
lunrItemsjs.prototype.reset = function () {
  this.emptySearchQuery();
  this.search();
};

/**
 * Build the lookup table for the filters
 */
lunrItemsjs.prototype.preProcessDocuments = function () {
  const filters = Object.keys(this.itemsjsConfiguration.aggregations);
  this.documents.forEach((doc, index) => {
    filters.forEach(filterId => {
      const filterKey = Object.keys(doc[filterId])[0];
      const filterVal = Object.values(doc[filterId])[0];
      if (filterVal !== '') {
        if (!this.lookups.hasOwnProperty(filterId)) {
          this.lookups[filterId] = [];
        }
        if (!this.lookups[filterId].hasOwnProperty(filterKey)) {
          this.lookups[filterId][filterKey] = filterVal;
        }
        this.documents[index][filterId] = filterKey
      }
      else {
        delete this.documents[index][filterId];
      }

    })
  });
};

lunrItemsjs.prototype.initSlugLookups = function () {
  if (!this.lookups.hasOwnProperty('slugs')) {
    this.lookups.slugs = [];
  }
  this.lookups.slugs['search'] = Drupal.t('search');
  this.lookups.slugs['currentPage'] = Drupal.t('page');
  Object.keys(this.filters).forEach(filter => {
    this.lookups.slugs[this.filters[filter]['field']] = this.filters[filter]['title_slug'];
  });
};

/**
 * Load the data from the endpoint.
 */
lunrItemsjs.prototype.loadData = function () {
  this.isLoading = true;
  const _this = this;
  const xhr = new XMLHttpRequest();
  xhr.open('GET', this.documentSourceEndoint);
  xhr.responseType = 'json';
  xhr.onload = function () {
    if (this.readyState === 4) {
      _this.documents = this.response;
      _this.initView();
    }
    else {
      _this.isLoading = false;
      _this.hasError = true;
    }
  };
  xhr.send();
};


document.addEventListener('alpine:init', () => {
  const lunrItemsjsInstance = new lunrItemsjs(drupalSettings.lunrItemsjsSettings);
  Alpine.magic('dbg', (el) => {
    return function (...args) {
      const raw = args.map(arg => Alpine.raw(arg));
      console.log(...raw);
    };
  });
  Alpine.data('lunrItemsjsView', () => ({
      lunrItemsjs: lunrItemsjsInstance,
    })
  );
});




