const lunrItemsjs = function (settings) {
  this.documentSourceEndoint = settings.source;
  this.language = settings.language;
  this.strings = {...settings.strings || {}};
  this.emptyResultString = 'No results found.'; // Not sure to keep this

  // Mostly Lunr settings

  // The Facet filters
  this.filters = settings.filters;
  // Lunr Searchable fields
  this.indexedFields = settings.indexed_fields;
  // Available sort options
  this.sorts = settings.sorts;
  // Primary key of indexed items, hardcoded in the view.
  this.indexPrimaryKey = settings.index_primary ?? '_itemsjs_id';

  // Item used for rendering results can be string or template
  this.listItem = settings.list_item;

  /**
   * Lookup table for internal usage.
   * @type {*[]}
   */
  this.lookups = [];

  this.activeFilters = [];
  this.totalResults = -1;


  /**
   * Loading state
   * @type {boolean}
   */
  this.isLoading = true;
  /**
   * Has load error?
   * @type {boolean}
   */
  this.hasError = false;

  this.itemsPerPage = settings?.per_page ?? -1;

  /**
   *  Indexable documents, loaded from the the documentSourceEndoint
   * @type {*[]}
   */
  this.documents = [];
  /**
   * Results for this page after a search has been performed.
   * @type { {} }
   */
  this.searchResults = {};

  /**
   * State of the current 'search'
   * @type {{filters: *[], search: string, currentPage: number}}
   */
  this.searchQuery = {
    filters: [],
    search: '',
    currentPage: 1,
  };

  this.pageUrl = new URL(window.location.href);

  this.itemsjsConfiguration = {
    sortings: this.sorts,
    aggregations: this.filters,
    searchableFields: Object.keys(this.indexedFields),
    native_search_enabled: false,
    custom_id_field: this.indexPrimaryKey
  };
  // Helper structure to make pagination easier.
  this.pagination = {
    total: this.totalResults,
    per_page: this.itemsPerPage,
    current_page: this.currentPage,
    last_page: Math.ceil(this.totalResults / this.itemsPerPage),
  };

  /**
   * Configuration for ItemsJS is rather static
   * @type {{searchableFields: string[], sortings, native_search_enabled:
   *   boolean, aggregations, custom_id_field: (*|string)}}
   */

  /**
   * Bind for fully indexed Lunr
   * @type {lunr|{}}
   */
  this.lunr = {};
  /**
   * Bind for fully indexed itemsjs
   * @type {itemsjs|{}}
   */
  this.itemsjs = {}; // Will hold the ItemsJS system

};

lunrItemsjs.prototype.initItemsJS = function () {

  this.itemsjs = itemsjs(this.documents, this.itemsjsConfiguration);
};


lunrItemsjs.prototype.initLunrSearch = function () {
  const _this = this;
  this.lunr = new lunr(function () {
    /**
     * The primary key for this index is `_itemsjs_id` as defined in the view
     * Make sure if you change that in a view, you change that in the
     * drupalSettings as well!
     *
     * Since users can add a custom field called ID it's set to this less
     * common guessable primary key by default.
     *
     * @see \Drupal\views_lunr_itemsjs\Plugin\search_api\backend\LunrBackend::indexItems
     */
    this.ref(_this.indexPrimaryKey);
    Object.keys(_this.indexedFields).map(field => {
      this.field(field);
    });
    _this.documents.forEach(function (document) {
      this.add(document);
    }, this);
  });
};

lunrItemsjs.prototype.initView = function () {

  this.preProcessDocuments();
  this.initLunrSearch();
  this.initItemsJS();
  this.emptySearch();
  this.search();
  this.isLoading = false;



};
/**
 * Returns a single active filter if available
 * @param filterName
 * @returns {*|boolean}
 */
lunrItemsjs.prototype.getActiveFilter = function (filterName) {
  return (this.activeFilters.hasOwnProperty(filterName)) ? this.activeFilters[filterName] : false;
};

lunrItemsjs.prototype.updatePagination = function () {
  this.pagination = {
    total: this.totalResults,
    per_page: this.itemsPerPage,
    current_page: this.currentPage,
    last_page: Math.ceil(this.totalResults / this.itemsPerPage),
  };
}

lunrItemsjs.prototype.search = function () {
  const search_result = this.lunr.search(this.searchQuery.search);
  let ids = search_result.map(v => v.ref);

  if (this.preFilter && this.preFilter instanceof Function){
    ids = this.preFilter(ids);
  }

  const itemsPerPage = (this.itemsPerPage === -1) ? this.documents.length : this.itemsPerPage;
  const searchQuery = {
    ids: ids,
    page: this.searchQuery.currentPage,
    per_page: itemsPerPage,
    sort: 'start_date',
    filters: this.searchQuery.filters,
  };

  const results = this.itemsjs.search(searchQuery);

  this.currentPage = results.pagination.page;
  this.totalResults = results.pagination.total;
  this.searchResults = results.data.items.map((result) => {
    return {listItem: result[this.listItem]};
  });
  this.activeFilters = results.data.aggregations;
  Object.keys(results.data.aggregations).map( filterId  =>{
    results.data.aggregations[filterId]['buckets'].map(bucket => {
      bucket['label'] = this.lookups[filterId][bucket['key']];
      return bucket;
    })
  })
  this.updatePagination();
};

lunrItemsjs.prototype.emptySearch = function () {
  this.searchQuery.search = '';
  this.currentPage = 1;
  Object.keys(this.filters).map(filter => {
    this.searchQuery.filters[filter] = [];
  });
};

/**
 * The click target for a reset button.
 */
lunrItemsjs.prototype.reset = function(){
  this.emptySearch();
  this.search();
};

/**
 * Build the lookup table for the filters
 */
lunrItemsjs.prototype.preProcessDocuments = function (){
  const filters = Object.keys(this.itemsjsConfiguration.aggregations);
  this.documents.forEach((doc, index) => {
    filters.forEach(filterId => {
      const filterKey = Object.keys(doc[filterId])[0];
      const filterVal = Object.values(doc[filterId])[0];
      if (filterVal !== '') {
        if (!this.lookups.hasOwnProperty(filterId)){
          this.lookups[filterId] =[];
        }
        if (!this.lookups[filterId].hasOwnProperty(filterKey)) {
          this.lookups[filterId][filterKey] = filterVal;
        }
        this.documents[index][filterId] = filterKey
      }
      else {
        delete this.documents[index][filterId];
      }

    })
  });
};

/**
 * Load the data from the endpoint.
 */
lunrItemsjs.prototype.loadData = function () {
  this.isLoading = true;
  const _this = this;
  const xhr = new XMLHttpRequest();
  xhr.open('GET', this.documentSourceEndoint);
  xhr.responseType = 'json';
  xhr.onload = function () {
    if (this.readyState === 4) {
      _this.documents = this.response;
      _this.initView();
    }
    else {
      _this.isLoading = false;
      _this.hasError = true;
    }
  };
  xhr.send();
};


document.addEventListener('alpine:init', () => {
  const lunrItemsjsInstance = new lunrItemsjs(drupalSettings.lunrItemsjsSettings);
  Alpine.magic('dbg', (el) => {
    return function (...args) {
      const raw = args.map(arg => Alpine.raw(arg));
      console.log(...raw);
    };
  });
  Alpine.data('lunrItemsjsView', () => ({
      lunrItemsjs: lunrItemsjsInstance,
    })
  );
});




