<?php

namespace Drupal\matone_vacancy\Importer\Operation;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\matone_vacancy\PipeInterface;

/**
 * Class SplitOperation
 *
 * Split the data in data that should be updated or data that should be created.
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class TranslateSplitOperation implements PipeInterface
{

	/**
	 * @var EntityStorageInterface
	 */
	protected $nodeStorage;

	/**
	 * @var LanguageManagerInterface
	 */
	protected $language_manager;

	/**
	 * Constructor.
	 *
	 * @param EntityTypeManagerInterface $entityTypeManager
	 *
	 * @param LanguageManagerInterface $language_manager
	 *
	 * @throws InvalidPluginDefinitionException
	 * @throws PluginNotFoundException
	 */
	public function __construct(
		EntityTypeManagerInterface $entityTypeManager,
		LanguageManagerInterface $language_manager
	) {
		$this->nodeStorage = $entityTypeManager->getStorage('node');
		$this->language_manager = $language_manager;
	}

	/**
	 * @inheritdoc
	 */
	public function handle($data, \Closure $next)
	{
		$vacancies = [];
		foreach ($data['data'] as $vacancy) {
			$vacancies[$vacancy['field_vacancy_id']][$vacancy['langcode']] = $vacancy;
		}
		$vacancies = array_map([$this, 'mapTranslatables'], $vacancies);
		$translations = $this->getTranslations($vacancies);
		$vacancies = array_map([$this, 'getDefaultVacancies'], $vacancies);

		$vacancy_ids = array_keys($vacancies);
		$update = [];
		if (count($vacancy_ids) > 0) {
			$existing_nodes = $this->nodeStorage->loadByProperties(['field_vacancy_id' => $vacancy_ids]);
			foreach ($existing_nodes as $node) {
				$id = $node->get('field_vacancy_id')->value;
				if (isset($vacancies[$id])) {
					$update[$id] = [
						'node' => $node,
						'data' => $vacancies[$id]
					];
					unset($vacancies[$id]);
				}
			}
		}

		$data['data'] = [
			'update' => $update,
			'create' => $vacancies,
			'translations' => $translations
		];

		return $next($data);
	}

	/**
	 * Get all the nodes that are translations.
	 *
	 * @param array $vacancies
	 *
	 * @return array
	 */
	private function getTranslations(array $vacancies)
	{
		$translations = [];
		foreach ($vacancies as $data) {
			if (isset($data['translations'])) {
				foreach ($data['translations'] as $translation) {
					array_push($translations, $translation);
				}
			}
		}
		return $translations;
	}

	/**
	 * Get the default vacancies.
	 *
	 * @param array $data
	 *
	 * @return array
	 */
	private function getDefaultVacancies(array $data)
	{
		if (isset($data['translations'])) {
			unset($data['translations']);
		}
		return reset($data);
	}

	/**
	 * Mark the vacancies that need translations as translations
	 *
	 * @param array $vacancies
	 *
	 * @return array
	 */
	private function mapTranslatables($vacancies)
	{
		$default_language = $this->language_manager->getDefaultLanguage();
		$default_langcode = $default_language->getId();

		if (isset($vacancies[$default_langcode])) {
			$translations_set = $vacancies;
			$default_vacancy = $translations_set[$default_langcode];
			unset($translations_set[$default_langcode]);

			if (count($translations_set)) {
				$vacancies = [
					$default_langcode => $default_vacancy,
					'translations' => $translations_set
				];
			}
		}
		return $vacancies;
	}
}
