<?php

namespace Drupal\matone_vacancy\Form;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\matone_vacancy\BackendInterface;
use Drupal\matone_vacancy\Plugin\BackendPluginManager;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SettingsForm
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class SettingsForm extends ConfigFormBase {

  /**
   * @var BackendPluginManager
   */
  private $backendPluginManager;

  /**
   * @var EntityStorageInterface
   */
  private $node_type_storage;

  /**
   * @var EntityStorageInterface
   */
  private $user_storage;

  /**
   * Constructor.
   *
   * @param ConfigFactoryInterface $config_factory
   * @param BackendPluginManager   $backend_plugin_manager
   * @param EntityStorageInterface $node_type_storage
   * @param EntityStorageInterface $user_storage
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    BackendPluginManager $backend_plugin_manager,
    EntityStorageInterface $node_type_storage,
    EntityStorageInterface $user_storage
  ) {
    parent::__construct($config_factory);
    $this->backendPluginManager = $backend_plugin_manager;
    $this->node_type_storage = $node_type_storage;
    $this->user_storage = $user_storage;
  }

  /**
   * @inheritdoc
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.backend'),
      $container->get('entity_type.manager')->getStorage('node_type'),
      $container->get('entity_type.manager')->getStorage('user')
    );
  }

  /**
   * @inheritdoc
   */
  protected function getEditableConfigNames() {
    return ['matone_vacancy.settings'];
  }

  /**
   * @inheritdoc
   */
  public function getFormId() {
    return 'matone_vacancy_settings';
  }

  /**
   * @inheritdoc
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('matone_vacancy.settings');

    $backendTypes = [];
    foreach ($this->backendPluginManager->getDefinitions() as $id => $definition) {
      $backendTypes[$id] = $definition['title'] ?? $id;
    }

    $current_backend = $config->get('backend_type');
    $form['backend_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Backend type'),
      '#options' => $backendTypes,
      '#default_value' => $current_backend,
      '#ajax' => [
        'trigger_as' => ['name' => 'backend_configure'],
        'callback' => '::buildAjaxBackendForm',
        'wrapper' => 'backend-ajax-form',
        'effect' => 'fade',
        'method' => 'replace',
        'event' => 'change'
      ]
    ];

    $form['backend_configure'] = [
      '#type' => 'submit',
      '#name' => 'backend_configure',
      '#value' => $this->t('Configure backend'),
      '#limit_validation_errors' => [['backend']],
      '#submit' => ['::submitAjaxBackendForm'],
      '#ajax' => [
        'callback' => '::buildAjaxBackendForm',
        'wrapper' => 'backend-ajax-form',
      ],
      '#attributes' => ['class' => ['js-hide']],
    ];

    $form['backend_config'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'backend-ajax-form'
      ]
    ];
    $this->buildBackendForm($form, $form_state);

    $form['mapping_config'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Mapping')
    ];
    $form['mapping_config']['bundle_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Bundle'),
      '#description' => $this->t('The bundle that the mapper should use for newly created nodes.'),
      '#options' => $this->getBundleOptions(),
      '#default_value' => $config->get('bundle_type') ?? NULL
    ];
    $form['mapping_config']['user'] = [
      '#type' => 'select',
      '#title' => $this->t('Content creator'),
      '#description' => $this->t('The user that is assigned the author for the newly created nodes.'),
      '#options' => $this->getUsers(),
      '#default_value' => $config->get('user') ?? 1
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * @inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $configured_backend = $form_state->getValue('backend_type');
    $config = $this->config('matone_vacancy.settings');
    $config
      ->set('backend_type', $configured_backend)
      ->set('bundle_type', $form_state->getValue('bundle_type'))
      ->set('user', $form_state->getValue('user'))
      ->save();
    
    if ($configured_backend) {
      /** @var BackendInterface $backend */
      $configuration_key = 'backend_config.' . $configured_backend;
      $previous_configuration = $config->get($configuration_key) ?? [];
      $backend = $this->backendPluginManager->createInstance($configured_backend, $previous_configuration);
      $subform_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);
      $backend->submitConfigurationForm($form['backend_config'], $subform_state);
      $config
        ->set($configuration_key, $backend->getConfiguration())
        ->save();
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Build a form based around a backend.
   *
   * @param array              $form
   * @param FormStateInterface $form_state
   *
   * @throws PluginException
   */
  private function buildBackendForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('matone_vacancy.settings');
    $backend = $config->get('backend_type');
    $inputs = $form_state->getUserInput();
    if (isset($inputs['backend_type'])) {
      $backend = $inputs['backend_type'];
    }

    if (!$backend) {
      return;
    }

    $backend_config = $config->get('backend_config.' . $backend) ?? [];

    /** @var BackendInterface $backend_type */
    $backend_type = $this->backendPluginManager->createInstance($backend, $backend_config);
    $subform_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);

    $type = 'fieldset';
    $config_form = $backend_type->buildConfigurationForm([], $subform_state);
    if (!$config_form) {
      $type = 'details';
      $config_form['#markup'] = $this->t('Backend type does not have a configuration form');
    }

    $form['backend_config'] = [
      '#type' => $type,
      '#tree' => TRUE,
      '#title' => $this->t('%backend settings', ['%backend' => $backend_type->getPluginDefinition()['title']]),
      '#attributes' => [
        'id' => 'backend-ajax-form'
      ]
    ] + $config_form;
  }

  /**
   * Submits ajax backend form.
   *
   * @param                    $form
   * @param FormStateInterface $form_state
   */
  public function submitAjaxBackendForm($form, FormStateInterface $form_state) {
    $form_state->setRebuild();
  }

  /**
   * Build a form after the user selects a different backend.
   *
   * @param array              $form
   * @param FormStateInterface $form_state
   *
   * @return mixed
   */
  public function buildAjaxBackendForm(array $form, FormStateInterface $form_state) {
    return $form['backend_config'];
  }

  /**
   * Get the bundle options for the mapping.
   *
   * @return array
   */
  private function getBundleOptions() {
    $node_types = $this->node_type_storage->loadByProperties([]);
    $result = [];
    foreach ($node_types as $node_type) {
      $result[$node_type->id()] = $node_type->label();
    }
    return $result;
  }

  /**
   * Get the available users in the system.
   *
   * @return array
   */
  private function getUsers() {
    /** @var User $users */
    $users = $this->user_storage->loadByProperties([]);
    $result = [];
    foreach ($users as $user) {
      $result[$user->id()] = $user->getDisplayName();
    }
    return $result;
  }
}
