<?php

namespace Drupal\matone_vacancy\Importer\Operation;

use Closure;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\matone_vacancy\PipeInterface;
use Drupal\matone_vacancy\ResultInterface;

/**
 * Class CleanUpOperation
 *
 * Clean the old vacancies from the database.
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class CleanUpOperation implements PipeInterface {

  /**
   * @var EntityStorageInterface
   */
  protected $nodeStorage;

  /**
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * Constructor.
   *
   * @param EntityTypeManagerInterface $entityTypeManager
   *
   * @throws InvalidPluginDefinitionException
   * @throws PluginNotFoundException
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager,ConfigFactoryInterface $configFactory) {
    $this->nodeStorage = $entityTypeManager->getStorage('node');
    $this->config = $configFactory->get('matone_vacancy.settings');
  }

  /**
   * @inheritdoc
   */
  public function handle($data, Closure $next) {
    /** @var ResultInterface $result */
    $result = $data['result'];

    $nodes = array_merge($result->getUpdates(), $result->getCreated());
    $node_ids = array_map(function ($node) {
      return $node->id();
    }, $nodes);

    $should_delete = $this->getNodes($node_ids);
    $this->nodeStorage->delete($should_delete);
    $result->setDeleted($should_delete);
    return $next($data);
  }

  /**
   * Find nodes that should be deleted.
   *
   * @param array $ids
   *     The node ids that are used to find nodes that should be deleted.
   *
   * @return array
   */
  private function getNodes(array $ids) {
    if (empty($ids)) {
      return [];
    }
    $bundleType = $this->config->get('bundle_type') ?? 'vacancy';
    $query = $this->nodeStorage->getQuery('AND')
      ->condition('type', $bundleType)
      ->condition('nid', $ids, 'NOT IN');
    $node_ids = $query->execute();
    return $this->nodeStorage->loadMultiple($node_ids);
  }
}
