<?php

namespace Drupal\matone_vacancy\Plugin\Mapping;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\matone_vacancy\Plugin\MappingBase;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @Mapping(
 *   id = "taxonomy_machine_name_mapping"
 * )
 * This plugin excpects that the taxonomy_machine_name module is active!
 */
class TaxonomyMachineNameMapping extends MappingBase {

    /**
     * @var EntityStorageInterface
     */
    private $term_storage;

    /**
     * @var Term[]
     */
    private $term_cache;

    /**
     * @var LanguageManagerInterface
     */
    private $language_manager;

    /**
     * Constructor.
     *
     * @param array                    $configuration
     * @param                          $plugin_id
     * @param                          $plugin_definition
     * @param EntityStorageInterface   $term_storage
     * @param LanguageManagerInterface $language_manager
     */
    public function __construct(
        array $configuration,
        $plugin_id,
        $plugin_definition,
        EntityStorageInterface $term_storage,
        LanguageManagerInterface $language_manager
    ) {
        parent::__construct($configuration, $plugin_id, $plugin_definition);
        $this->term_storage = $term_storage;
        $this->language_manager = $language_manager;

        $this->prepareTermCache();
    }

    /**
     * @inheritdoc
     */
    public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
        return new static(
            $configuration,
            $plugin_id,
            $plugin_definition,
            $container->get('entity_type.manager')->getStorage('taxonomy_term'),
            $container->get('language_manager')
        );
    }

    /**
     * @inheritdoc
     */
    public function map($data) {
        if (!$data){
            return;
        }
        [$vid, $values] = array_values($data);
        if (!is_array($values)) {
            $values = [$values];
        }

        $result = [];
        foreach ($values as $value) {
            if (!$value) {
                continue;
            }
           $machineName = taxonomy_machine_name_clean_name($value);
           $term = $this->getFromCache($vid, $machineName);
            if (!$term) {
                $term_data = [
                    'vid' => $vid,
                    'name' => $value,
                    'langcode' => $this->language_manager->getDefaultLanguage()->getId()
                ];
                $term = Term::create($term_data);
                try {
                    $term->save();
                } catch (EntityStorageException $e) {
                    watchdog_exception('rz vacancy', $e, 'Taxonomy Mapping: %type: @message in %function (line %line of %file).');
                    continue;
                }
                $this->term_cache[$vid][$term->label()] = $term;
            }
            $result[] = $term->id();
        }

        return $result;
    }

    /**
     * Prepare the taxonomy term cache with the terms currently in the database.
     *
     * @return void
     */
    private function prepareTermCache() {
        $tids = $this->term_storage->getQuery()->execute();
        $terms = $this->term_storage->loadMultiple($tids);

        /** @var Term $term */
        foreach ($terms as $term) {
            $machineName = $term->get('machine_name')->first()->getString();
            $this->term_cache[$term->bundle()][$machineName] = $term;
        }
    }

    /**
     * Get taxonomy term from cache.
     *
     * @param $vid
     * @param $value
     *
     * @return Term|null
     */
    private function getFromCache($vid, $value) {
        return $this->term_cache[$vid][$value] ?? NULL;
    }
}
