<?php

namespace Drupal\matone_vacancy\Plugin\Mapping;

use Drupal\Core\Entity\EntityStorageException;
use Drupal\matone_enrichment\Entity\Enrichment;
use Drupal\matone_enrichment\Entity\EnrichmentInterface;
use Drupal\matone_enrichment\EnrichmentStorageInterface;
use Drupal\matone_vacancy\Plugin\MappingBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * @Mapping(
 *   id = "enrichment_mapping"
 * )
 */
class EnrichmentMapping extends MappingBase {

	/**
	 * @var EnrichmentStorageInterface
	 */
	private $enrichment_storage;

	/**
	 * @var EnrichmentInterface[]
	 */
	private $enrichment_cache = [];

	/**
	 * Constructor.
	 *
	 * @param array                      $configuration
	 * @param                            $plugin_id
	 * @param                            $plugin_definition
	 * @param EnrichmentStorageInterface $enrichment_storage
	 */
	public function __construct(
		array $configuration,
		$plugin_id,
		$plugin_definition,
		EnrichmentStorageInterface $enrichment_storage
	) {
		parent::__construct($configuration, $plugin_id, $plugin_definition);
		$this->enrichment_storage = $enrichment_storage;

		$this->prepareCache();
	}

	/**
	 * @inheritdoc
	 */
	public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
		return new static(
			$configuration,
			$plugin_id,
			$plugin_definition,
			$container->get('entity_type.manager')->getStorage('enrichment')
		);
	}


	/**
	 * @inheritdoc
	 */
	public function map($data) {
		if (!\is_array($data) || !isset($data['value'])) {
			return NULL;
		}

		$function_group = $data['value'];
		$machine_name = $this->buildMachineName($function_group);

		$enrichment = $this->getEnrichment($machine_name);
		if (!$enrichment) {
			try {
				$enrichment = $this->createEnrichment($function_group, $machine_name);
			} catch (EntityStorageException $e) {
				watchdog_exception('vacancy', $e, 'Enrichment Mapping: %type: @message in %function (line %line of %file).');
				return NULL;
			}
		}

		return $enrichment->id();
	}

	/**
	 * Get the enrichment list.
	 *
	 * @param string $machine_name
	 *
	 * @return EnrichmentInterface|null
	 */
	private function getEnrichment(string $machine_name) {
		return $this->enrichment_cache[$machine_name] ?? NULL;
	}

	/**
	 * Create an enrichment object.
	 *
	 * @param string $enrichment_name
	 * @param string $machine_name
	 *
	 * @return \Drupal\Core\Entity\EntityInterface
	 * @throws EntityStorageException
	 */
	private function createEnrichment(string $enrichment_name, string $machine_name) {
		$config = \Drupal::configFactory();
		$user = $config->get('matone_vacancy.settings')->get('user');
		$enrichment_data = [
			'label'   => $enrichment_name,
			'name'    => $machine_name,
			'user_id' => 1,
			'field_generated' => $user
		];

		/** @var Enrichment $enrichment */
		$enrichment = $this->enrichment_storage->create($enrichment_data);
		$enrichment->save();

		$this->enrichment_cache[$enrichment->getName()] = $enrichment;

		return $enrichment;
	}


	/**
	 * Prepare the enrichment cache.
	 *
	 * @return void
	 */
	private function prepareCache() {
		$entity_ids = $this->enrichment_storage->getQuery()->execute();
		$entities = $this->enrichment_storage->loadMultiple($entity_ids);

		/** @var Enrichment $entity */
		foreach ($entities as $entity) {
			$this->enrichment_cache[$entity->getName()] = $entity;
		}
	}
}
