<?php

namespace Drupal\matone_vacancy;

use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class Pipeline
 *
 * This is a simple version based on the Laravel pipeline.
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class Pipeline {

  /**
   * @var array
   */
  protected $pipes = [];

  /**
   * @var string
   */
  protected $method = 'handle';

  /**
   * @var mixed
   */
  protected $passable;

  /**
   * @var ContainerInterface
   */
  protected $container;

  /**
   * Constructor.
   *
   * @param ContainerInterface $container
   */
  public function __construct(ContainerInterface $container) {
    $this->container = $container;
  }

  /**
   * Set the object that should be passed.
   *
   * @param $passable
   *
   * @return $this
   */
  public function send($passable) {
    $this->passable = $passable;
    return $this;
  }

  /**
   * Set the array of pipes.
   *
   * @param array $pipes
   *
   * @return Pipeline
   */
  public function through($pipes) {
    $this->pipes = $pipes;
    return $this;
  }

  /**
   * Set the method to call on the pipes.
   *
   * @param string $method
   *
   * @return Pipeline
   */
  public function via($method) {
    $this->method = $method;
    return $this;
  }

  /**
   * Run the pipeline with a destination.
   *
   * @param $destination
   *
   * @return mixed
   */
  public function then($destination) {
    $pipeline = array_reduce(
      array_reverse($this->pipes()), $this->carry(), $this->prepareDestination($destination)
    );
    return $pipeline($this->passable);
  }

  /**
   * Run the pipeline and return the result.
   *
   * @return mixed
   */
  public function thenReturn() {
    return $this->then(function ($passable) {
      return $passable;
    });
  }

  /**
   * Get a closure that represents a slice of the pipeline stack.
   *
   * @return \Closure
   */
  public function carry() {
    return function ($stack, $pipe) {
      return function ($passable) use ($stack, $pipe) {
        if (is_callable($pipe)) {
          return $pipe($passable, $stack);
        }

        if ($this->container->has($pipe)) {
          $object = $this->container->get($pipe) ?? '';
        } else {
          $object = new $pipe();
        }

        $carry = method_exists($object, $this->method)
          ? $object->{$this->method}($passable, $stack)
          : $object($passable, $stack);
        return $carry;
      };
    };
  }

  /**
   * Prepare the destination function to call after the pipeline is finished running.
   *
   * @param $destination
   *
   * @return \Closure
   */
  public function prepareDestination($destination) {
    return function ($passable) use ($destination) {
      return $destination($passable);
    };
  }

  /**
   * Return the array of pipes.
   *
   * @return array
   */
  public function pipes() {
    return $this->pipes;
  }
}
