<?php

namespace Drupal\matone_vacancy\Commands;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\matone_vacancy\BackendInterface;
use Drupal\matone_vacancy\Importer\VacancyImporter;
use Drupal\matone_vacancy\ImportResult;
use Drupal\matone_vacancy\Plugin\BackendPluginManager;
use Drupal\matone_vacancy\ResultInterface;
use Drush\Commands\DrushCommands;
use Drush\Utils\StringUtils;
use Psr\Log\LoggerInterface;

/**
 * Class VacancyCommands
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class VacancyCommands extends DrushCommands {

  /**
   * @var ConfigFactoryInterface
   */
  private $configFactory;

  /**
   * @var BackendPluginManager
   */
  private $backendPluginManager;

  /**
   * @var VacancyImporter
   */
  private $vacancyImporter;
  /**
   * @var LoggerInterface
   */
  private $import_logger;

  /**
   * Constructor.
   *
   * @param ConfigFactoryInterface $config_factory
   * @param BackendPluginManager   $backend_plugin_manager
   * @param VacancyImporter        $vacancy_importer
   * @param LoggerInterface        $import_logger
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    BackendPluginManager $backend_plugin_manager,
    VacancyImporter $vacancy_importer,
    LoggerInterface $import_logger
  ) {
    $this->backendPluginManager = $backend_plugin_manager;
    $this->vacancyImporter = $vacancy_importer;
    $this->configFactory = $config_factory;
    $this->import_logger = $import_logger;
  }

  /**
   * Import vacancies.
   *
   * @command matone:import:vacancies
   * @aliases miv,matone-import-vacancies
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function importVacancies() {
    $config = $this->configFactory->get('matone_vacancy.settings');
    $backend_type = $config->get('backend_type');
    $backend_config = $config->get('backend_config.' . $backend_type) ?? [];

    /** @var BackendInterface $backend */
    $backend = $this->backendPluginManager->createInstance($backend_type, $backend_config);
    $vacancies = $backend->getVacancies();
    $result = $this->vacancyImporter->import($vacancies);

    $output = [
      'updated' => $result->getUpdates(),
      'created' => $result->getCreated(),
      'deleted' => $result->getDeleted(),
      'failed' => $result->getFailed()
    ];

    foreach ($output as $title => $data) {
      foreach ($data as $node) {
        $this->writeln(
          StringUtils::interpolate(
            '<info>@operation: @title (id: @nid)</info>',
            ['@operation' => $title, '@title' => $node->getTitle(), '@nid' => $node->id()]
          )
        );
      }
    }

    $message = t(
      'Import succeeded. Created @created, Updated @updated, Removed @removed, Failed @failed',
      [
        '@created' => count($output['created']),
        '@updated' => count($output['updated']),
        '@removed' => count($output['deleted']),
        '@failed' => count($output['failed'])
      ]
    );

    $this->import_logger->info($message);
  }
}
