<?php

namespace Drupal\matone_vacancy\Importer;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\matone_vacancy\MappableObject;
use Drupal\matone_vacancy\MappingInterface;
use Drupal\matone_vacancy\Plugin\MappingPluginManager;

/**
 * Class Mapper
 *
 * Maps data from a MappableObject to the fields defined in the MappableObject.
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class Mapper {

  /**
   * @var MappingPluginManager
   */
  private $mappingPluginManager;

  /**
   * @var MappingInterface[]
   */
  private $mappingCache;

  /**
   * @var string[]
   */
  private $methodCache;

  /**
   * @var ConfigFactoryInterface
   */
  private $config_factory;

  /**
   * Constructor.
   *
   * @param MappingPluginManager   $mappingPluginManager
   * @param ConfigFactoryInterface $config_factory
   */
  public function __construct(MappingPluginManager $mappingPluginManager, ConfigFactoryInterface $config_factory) {
    $this->mappingPluginManager = $mappingPluginManager;
    $this->config_factory = $config_factory;
  }

  /**
   * Converts a mappable object to an array.
   *
   * @param MappableObject $object
   *
   * @return array
   * @throws \Exception
   */
  public function map(MappableObject $object) {
    $mapping = $object->getMapping();

    $data = [];
    foreach ($mapping as $field_name => $info) {
      $mapper = $this->getMapping($info['mapper']);
      $fields = [];
      foreach ($info['fields'] as $field) {
        if (!isset($this->methodCache[$field])) {
          $this->methodCache[$field] = 'get' . str_replace('_', '', ucwords($field, '_'));
        }
        $method = $this->methodCache[$field];
        $fields[$field] = $object->{$method}();
      }
      $data[$field_name] = $mapper->map(count($fields) === 1 ? reset($fields) : $fields);
    }
    $data = $data + $this->getDefaultValues();
    return $data;
  }

  /**
   * Get default values.
   *
   * @return array
   */
  private function getDefaultValues() {
    $config = $this->config_factory->get('matone_vacancy.settings');

    return [
      'type' => $config->get('bundle_type') ?? 'vacancy',
      'uid'  => $config->get('user') ?? 1
    ];
  }

  /**
   * Get mapping from cache.
   * If it does not exist resolve the name and create one.
   *
   * @param string $mapping
   *
   * @return MappingInterface
   * @throws PluginException
   */
  private function getMapping(string $mapping) {
    if (!isset($this->mappingCache[$mapping])) {
      $this->mappingCache[$mapping] = $this->mappingPluginManager->createInstance($mapping);
    }
    return $this->mappingCache[$mapping];
  }
}
