<?php

namespace Drupal\matone_vacancy\Importer;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\matone_vacancy\MappableObject;
use Drupal\matone_vacancy\MappingInterface;
use Drupal\matone_vacancy\Plugin\MappingPluginManager;

/**
 * Class Mapper
 *
 * Maps data from a MappableObject to the fields defined in the MappableObject.
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class Mapper {

  /**
   * @var MappingPluginManager
   */
  private $mappingPluginManager;

  /**
   * @var MappingInterface[]
   */
  private $mappingCache;

  /**
   * @var string[]
   */
  private $methodCache;

  /**
   * Constructor.
   *
   * @param MappingPluginManager $mappingPluginManager
   */
  public function __construct(MappingPluginManager $mappingPluginManager) {
    $this->mappingPluginManager = $mappingPluginManager;
  }

  /**
   * Converts a mappable object to an array.
   *
   * @param MappableObject $object
   *
   * @return array
   * @throws \Exception
   */
  public function map(MappableObject $object) {
    $mapping = $object->getMapping();

    $data = [];
    foreach ($mapping as $field_name => $info) {
      $mapper = $this->getMapping($info['mapper']);
      $fields = [];
      foreach ($info['fields'] as $field) {
        if (!isset($this->methodCache[$field])) {
          $this->methodCache[$field] = 'get' . str_replace('_', '', ucwords($field, '_'));
        }
        $method = $this->methodCache[$field];
        $fields[$field] = $object->{$method}();
      }
      $data[$field_name] = $mapper->map(count($fields) === 1 ? reset($fields) : $fields);
    }
    $data = $data + $this->getDefaultValues();
    return $data;
  }

  /**
   * Get default values.
   *
   * @return array
   */
  private function getDefaultValues() {
    return [
      'type' => 'vacancy',
      'uid' => 1
    ];
  }

  /**
   * Get mapping from cache.
   * If it does not exist resolve the name and create one.
   *
   * @param string $mapping
   *
   * @return MappingInterface
   * @throws PluginException
   */
  private function getMapping(string $mapping) {
    if (!isset($this->mappingCache[$mapping])) {
      $this->mappingCache[$mapping] = $this->mappingPluginManager->createInstance($mapping);
    }
    return $this->mappingCache[$mapping];
  }
}
