<?php

namespace Drupal\matone_vacancy\Form;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\matone_vacancy\BackendInterface;
use Drupal\matone_vacancy\Plugin\BackendPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SettingsForm
 *
 * @author Sander Boelhouwers <sander.boelhouwers@ara.nl>
 */
class SettingsForm extends ConfigFormBase {

  /**
   * @var BackendPluginManager
   */
  private $backendPluginManager;

  /**
   * Constructor.
   *
   * @param ConfigFactoryInterface $config_factory
   * @param BackendPluginManager   $backend_plugin_manager
   */
  public function __construct(ConfigFactoryInterface $config_factory, BackendPluginManager $backend_plugin_manager) {
    parent::__construct($config_factory);
    $this->backendPluginManager = $backend_plugin_manager;
  }

  /**
   * @inheritdoc
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('plugin.manager.backend')
    );
  }

  /**
   * @inheritdoc
   */
  protected function getEditableConfigNames() {
    return ['matone_vacancy.settings'];
  }

  /**
   * @inheritdoc
   */
  public function getFormId() {
    return 'matone_vacancy_settings';
  }

  /**
   * @inheritdoc
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('matone_vacancy.settings');

    $backendTypes = [];
    foreach ($this->backendPluginManager->getDefinitions() as $id => $definition) {
      $backendTypes[$id] = $definition['title'] ?? $id;
    }

    $current_backend = $config->get('backend_type');
    $form['backend_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Backend type'),
      '#options' => $backendTypes,
      '#default_value' => $current_backend,
      '#ajax' => [
        'trigger_as' => ['name' => 'backend_configure'],
        'callback' => '::buildAjaxBackendForm',
        'wrapper' => 'backend-ajax-form',
        'effect' => 'fade',
        'method' => 'replace',
        'event' => 'change'
      ]
    ];

    $form['backend_configure'] = [
      '#type' => 'submit',
      '#name' => 'backend_configure',
      '#value' => $this->t('Configure backend'),
      '#limit_validation_errors' => [['backend']],
      '#submit' => ['::submitAjaxBackendForm'],
      '#ajax' => [
        'callback' => '::buildAjaxBackendForm',
        'wrapper' => 'backend-ajax-form',
      ],
      '#attributes' => ['class' => ['js-hide']],
    ];

    $form['backend_config'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => 'backend-ajax-form'
      ]
    ];
    $this->buildBackendForm($form, $form_state);

    return parent::buildForm($form, $form_state);
  }

  /**
   * @inheritdoc
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $configured_backend = $form_state->getValue('backend_type');
    $config = $this->config('matone_vacancy.settings');
    $config
      ->set('backend_type', $configured_backend)
      ->save();
    
    if ($configured_backend) {
      /** @var BackendInterface $backend */
      $configuration_key = 'backend_config.' . $configured_backend;
      $previous_configuration = $config->get($configuration_key) ?? [];
      $backend = $this->backendPluginManager->createInstance($configured_backend, $previous_configuration);
      $subform_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);
      $backend->submitConfigurationForm($form['backend_config'], $subform_state);
      $config
        ->set($configuration_key, $backend->getConfiguration())
        ->save();
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Build a form based around a backend.
   *
   * @param array              $form
   * @param FormStateInterface $form_state
   *
   * @throws PluginException
   */
  private function buildBackendForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('matone_vacancy.settings');
    $backend = $config->get('backend_type');
    $inputs = $form_state->getUserInput();
    if (isset($inputs['backend_type'])) {
      $backend = $inputs['backend_type'];
    }

    if (!$backend) {
      return;
    }

    $backend_config = $config->get('backend_config.' . $backend) ?? [];

    /** @var BackendInterface $backend_type */
    $backend_type = $this->backendPluginManager->createInstance($backend, $backend_config);
    $subform_state = SubformState::createForSubform($form['backend_config'], $form, $form_state);

    $type = 'fieldset';
    $config_form = $backend_type->buildConfigurationForm([], $subform_state);
    if (!$config_form) {
      $type = 'details';
      $config_form['markup'] = $this->t('Backend type does not have a configuration form');
    }

    $form['backend_config'] = [
      '#type' => $type,
      '#tree' => TRUE,
      '#title' => $this->t('%backend settings', ['%backend' => $backend_type->getPluginDefinition()['title']]),
      '#attributes' => [
        'id' => 'backend-ajax-form'
      ]
    ] + $config_form;
  }

  /**
   * Submits ajax backend form.
   *
   * @param                    $form
   * @param FormStateInterface $form_state
   */
  public function submitAjaxBackendForm($form, FormStateInterface $form_state) {
    $form_state->setRebuild();
  }

  /**
   * Build a form after the user selects a different backend.
   *
   * @param array              $form
   * @param FormStateInterface $form_state
   *
   * @return mixed
   */
  public function buildAjaxBackendForm(array $form, FormStateInterface $form_state) {
    return $form['backend_config'];
  }
}
