<?php declare(strict_types=1);

namespace Drupal\matone_helpers;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ConfigInstallerInterface;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Config\InstallStorage;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Extension\ModuleInstallerInterface;
use Drupal\Core\File\FileSystemInterface;
use Symfony\Component\Yaml\Yaml;


class InstallerService {

  protected ModuleInstallerInterface $moduleInstaller;

  protected ModuleHandlerInterface $moduleHandler;

  protected ConfigInstallerInterface $configInstaller;

  protected FileSystemInterface $fileSystem;

  protected ConfigFactoryInterface $configFactory;

  protected string $moduleName;

  protected string $moduleInfoFile;

  protected string $optionalInstallPath;

  public function __construct(ModuleInstallerInterface $moduleInstaller,
                              ModuleHandlerInterface $moduleHandler,
                              ConfigInstallerInterface $configInstaller,
                              FileSystemInterface $fileSystem,
                              ConfigFactoryInterface $configFactory
  ) {
    $this->moduleInstaller = $moduleInstaller;
    $this->moduleHandler = $moduleHandler;
    $this->configInstaller = $configInstaller;
    $this->fileSystem = $fileSystem;
    $this->configFactory = $configFactory;
  }


  /**
   * The only function you should call
   */
  public function installForModule(string $moduleName): void {
  if (\Drupal::isConfigSyncing()) {
      return;
    }
    if (FALSE !== $this->initModulePaths($moduleName)) {
      $this->enableDefinedModules();
      $this->installConfiguration();
    }
  }

  protected function initModulePaths(string $moduleName): bool {
    $moduleName = preg_replace('/_install$/', '', $moduleName);
    if (NULL !== $moduleName) {
      \Drupal::logger('matone_helpers')
        ->info('Started install of ' . $moduleName);
      $this->moduleName = $moduleName;
      $path = $this->moduleHandler->getModule($this->moduleName)->getPath();
      $this->moduleInfoFile = sprintf('%s/%s.info.yml', $path, $this->moduleName);
      $this->optionalInstallPath = sprintf('%s/%s', $path, InstallStorage::CONFIG_OPTIONAL_DIRECTORY);
      return file_exists($this->moduleInfoFile);
    }
    return FALSE;
  }

  protected function enableDefinedModules(): void {
    $moduleInfoContent = (array) Yaml::parse(file_get_contents($this->moduleInfoFile));
    if (isset($moduleInfoContent['install']) && is_array($moduleInfoContent['install'])) {
      $this->moduleInstaller->install($moduleInfoContent['install'], TRUE);
    }
  }

  protected function installConfiguration(): void {
    if (is_dir($this->optionalInstallPath)) {
      // First install the default configuration
      $this->configInstaller->installDefaultConfig('module', $this->moduleName);
      // Then create the field storage configuration
      $this->createFieldStorageConfiguration();
      // Then install the rest of the optional config;
      $this->installOptionalConfiguration();
      // And finnaly force save the *.settings.yml files from the optional config
      $this->createSettingsConfiguration();
    }
  }

  // Create field storage configs in the active configuration

  protected function createFieldStorageConfiguration(): void {
    $storageConfigFiles = $this->fileSystem->scanDirectory($this->optionalInstallPath, '/^field.storage.*\\.(yml)$/i');
    $this->parseAndInstallConfigFiles($storageConfigFiles);
  }

  protected function parseAndInstallConfigFiles(array $settingsConfigFiles): void {
    if (isset($settingsConfigFiles) && is_array($settingsConfigFiles)) {
      foreach ($settingsConfigFiles as $settingsConfigFile) {
        $settingsFileContent = file_get_contents(DRUPAL_ROOT . '/' . $settingsConfigFile->uri);
        $parsedSettings = (array) Yaml::parse($settingsFileContent);
        $config_factory = $this->configFactory->getEditable($settingsConfigFile->name);
        $config_factory->setData($parsedSettings)->save(TRUE);
      }
    }
  }


  protected function installOptionalConfiguration(): void {
    $storage = new FileStorage($this->optionalInstallPath, StorageInterface::DEFAULT_COLLECTION);
    $this->configInstaller->installOptionalConfig($storage, '');
  }

  protected function createSettingsConfiguration(): void {
    $settingsConfigFiles = $this->fileSystem->scanDirectory($this->optionalInstallPath, '/^.*\\.(settings.yml)$/i');
    $this->parseAndInstallConfigFiles($settingsConfigFiles);
  }

}
