<?php

namespace Drupal\matone_content_block\Plugin\Field\FieldFormatter;

use Drupal\Component\Utility\Html;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element;

/**
 * Plugin implementation of the 'block_content_reference_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "block_content_reference_formatter",
 *   label = @Translation("Block content as block"),
 *   field_types = {
 *     "entity_reference"
 *   }
 * )
 */
class BlockContentReferenceFormatter extends EntityReferenceEntityFormatter implements ContainerFactoryPluginInterface
{


  public function settingsForm(array $form, FormStateInterface $form_state)
  {
    $view_modes = $this->getConfigurableViewModes();
    if (substr_count(!$this->getFieldSetting('handler'), 'block_content')) {
      $form['warning'] = [
        '#type' => 'html_tag',
        '#tag' => 'p',
        '#value' => $this->t('This formatter can only be used for block_content references!'),
      ];
    } elseif (!empty($view_modes)) {
      $form['view_mode'] = [
        '#title' => t('View Mode'),
        '#description' => t('Select the view mode to use for rendering'),
        '#type' => 'select',
        '#default_value' => $this->getSettings()['view_mode'],
        '#options' => $view_modes,
      ];
    }
    return $form;
  }

  public function settingsSummary()
  {
    if (substr_count(!$this->getFieldSetting('handler'), 'block_content')) {
      return ['#markup' => $this->t('This formatter can only be used for <b>block_content</b>references!')];
    }
    return ['#markup' => $this->t('Block(s) rendered in "@view_mode" view-mode.', ['@view_mode' => $this->getSetting('view_mode')])];

  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode)
  {

    $view_mode = $this->getSetting('view_mode');
    $elements = [];

    foreach ($this->getEntitiesToView($items, $langcode) as $delta => $entity) {

      $view_builder = $this->entityTypeManager->getViewBuilder($entity->getEntityTypeId());
      $view = $view_builder->view($entity, $view_mode, $entity->language()->getId());
      $content = [];
      foreach (Element::children($view) as $key) {
        $content['content'][$key] = $view[$key];
        unset($view[$key]);
      }
      $view['#pre_render'] = [];
      $elements[$delta] = array_merge($view,$content);
      $elements[$delta]['#cache'] = $view['#cache'];
      $elements[$delta]['#theme'] = 'block_content';
    }

    return $elements;
  }

  /**
   * @return array
   */
  protected function getConfigurableViewModes()
  {
    return $this->entityDisplayRepository->getViewModeOptions($this->getTargetEntityId($this->fieldDefinition));
  }

  /**
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *
   * @return mixed
   */
  protected function getTargetEntityId(FieldDefinitionInterface $field_definition)
  {
    return $field_definition->getFieldStorageDefinition()
      ->getSetting('target_type');
  }

  protected function setDefaultAttributes($id, array &$config) {
    if (!isset($config['#attributes']['id'])) {
      $config['#attributes']['id'] = Html::getUniqueId('block-' . $id);
    }
    $config['#attributes']['class'][] = Html::cleanCssIdentifier('block-' . $id);
  }


  public static function defaultSettings()
  {
    return [
      'view_mode' => 'default',
    ];
  }
}
