<?php

namespace Drupal\Matone\Plugin\Layout;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Layout\LayoutDefault;
use Drupal\Core\Plugin\PluginFormInterface;


class Layout extends LayoutDefault implements PluginFormInterface {


  protected $isEditing = FALSE;

  public function __construct(array $configuration, $plugin_id, $plugin_definition) {
    $this->isEditing = \Drupal::service('router.admin_context')->isAdminRoute();
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }


  /**
   * @inheritDoc
   */
  public function defaultConfiguration() {
    $configuration = parent::defaultConfiguration();
    return $configuration + [
        'anchor' => '',
      ];
  }

  /**
   * @inheritDoc
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['anchor'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Anchor link'),
      '#default_value' => $this->configuration['anchor'],
      '#description' => $this->t('Add anchor name in lowercase and no spaces. You can use this name as a reference to jump to content.'),
    ];
    return parent::buildConfigurationForm($form, $form_state);
  }

  /**
   * @inheritDoc
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {

  }

  /**
   * @inheritDoc
   */
  public function build(array $regions): array {
    $build = parent::build($regions);
    $build['#attributes']['class'] = [
      'layout',
      // This layout piggybacks the default layouts
      $this->getPluginDefinition()->getTemplate(),
    ];

    if ($this->isEditing) {
      $build['#attributes']['class'][] = "edit";
      $build['#attached']['library'][] = 'matone_layout/layout-admin-tweaks';
    }
    else {
      foreach ($this->getPluginDefinition()->getRegionNames() as $regionName) {
        if (array_key_exists($regionName, $build)) {
          array_walk($build[$regionName], [
            'self',
            'setParagraphsViewModeForRegion',
          ]);
        }
      }
    }
    return $build;
  }

  public function setParagraphsViewModeForRegion(&$paragraph) {
    switch ($this->pluginId) {
      case 'matone_layout_onecol' :
        $paragraph['#view_mode'] = 'onecol';
        break;
      case 'matone_layout_twocol' :
        $paragraph['#view_mode'] = 'twocol';
        break;
      case 'matone_layout_threecol' :
        $paragraph['#view_mode'] = 'threecol';
        break;
      case 'matone_layout_fourcol' :
        $paragraph['#view_mode'] = 'fourcol';
        break;
      default:
        $paragraph['#view_mode'] = 'full';
    }
  }

  /**
   * @inheritDoc
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    if (!$form_state->isValueEmpty('anchor')) {
      $form_state->setValue('anchor', $this->cleanAnchorName($form_state->getValue('anchor')));
    }
    $this->configuration['anchor'] = $form_state->getValue('anchor');
    parent::submitConfigurationForm($form, $form_state);
  }


  /**
   * Language aware lowercase the anchor link
   *
   * @param $name
   *
   * @return mixed|string
   */
  public function cleanAnchorName($name) {
    if (!preg_match('/^[a-z0-9\_]+$/', $name)) {
      $unknown_character = '_';
      // Transliterate and sanitize the destination filename.
      $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();
      $name = \Drupal::transliteration()
        ->transliterate($name, $langcode, $unknown_character);
      $name = mb_strtolower(trim($name));
      $name = trim(preg_replace('/[^a-z0-9\_]+/', $unknown_character, $name), $unknown_character);
    }
    return $name;
  }

}
