<?php

namespace Drupal\mapping_service\Attribute;

use Attribute;
use Drupal\mapping_service\Contract\EntityFieldInterface;
use InvalidArgumentException;

#[Attribute(Attribute::TARGET_PROPERTY | Attribute::IS_REPEATABLE)]
class EntityField {

  private static array $instanceCache = [];

  private string|EntityFieldInterface $fieldHandler;

  private string|array $targets;

  public function __construct(
    string       $type,
    string|array $targets = [],
    array        $options = []
  ) {
    $this->targets = $targets;
    $this->fieldHandler = $this->resolveHandler($type, $options);
  }

  // check if $type is class, do those first!
  private function resolveHandler(string $type, array $options): mixed {
    // Handle primitive types, expand and clean this
    if (in_array($type, [
      'string',
      'int',
      'float',
      'bool',
      'array',
      'timestamp',
    ])) {
      return $type;
    }
    if (class_exists($type) && is_a($type, EntityFieldInterface::class, TRUE)) {
      $cacheID = $this->getCacheKey($type, $options);
      if (isset(self::$instanceCache[$cacheID])) {
        return self::$instanceCache[$cacheID];
      }
      $instance = new $type(...$options);
      self::$instanceCache[$cacheID] = $instance;
      return $instance;
    }
    throw new \InvalidArgumentException("Invalid `$type`, class must implement EntityFieldInterface.");
  }

  private function getCacheKey(string $class, array $options): string {
    return $class . ':' . hash('xxh32', serialize($options));
  }

  public static function clearCache(): void {
    self::$instanceCache = [];
  }

  /**
   * @throws \DateMalformedStringException
   */
  public function getMappedValues(mixed $value): array {

    $fields = $this->getFields();

    // Handle primitive types
    if (is_string($this->fieldHandler)) {
      $castValue = match ($this->fieldHandler) {
        'string' => (string) $value,
        'int' => (int) $value,
        'float' => (float) $value,
        'bool' => (bool) $value,
        'array' => (array) $value,
        'timestamp' => (new \DateTime($value))->format('U'),
        default => throw new InvalidArgumentException("Unsupported type: {$this->fieldHandler}")
      };

      if (empty($castValue)) {
        return [];
      }
      $result = [];
      foreach ($fields as $field) {
        $result[$field] = $castValue;
      }
      return $result;
    }

    // Handle mapper class
    if (is_object($this->fieldHandler) && method_exists($this->fieldHandler, 'map')) {
      $mappedValue = $this->fieldHandler->map($value);

      if (empty($mappedValue)) {
        return [];
      }

      if (method_exists($this->fieldHandler, 'mapToFields')) {
        $result = $this->fieldHandler->mapToFields($fields);
      }
      else {
        $result = [];
        foreach ($fields as $field) {
          $result[$field] = $mappedValue;
        }
      }
      // Map values to fields in order

      return $result;
    }

    throw new InvalidArgumentException('Handler must be a primitive type or have map() method');
  }

  public function getFields(): array {
    return is_array($this->targets) ? $this->targets : [$this->targets];
  }

  public function getFieldHandler(): ?EntityFieldInterface {
    if ($this->fieldHandler && !is_string($this->fieldHandler)) {
      return $this->fieldHandler;
    }
    return NULL;
  }
}
