<?php

declare(strict_types=1);

namespace Drupal\mapping_service\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\Entity\User;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Attribute settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  protected EntityStorageInterface $userStorage;


  public function __construct(ConfigFactoryInterface $config_factory,  TypedConfigManagerInterface $typedConfigManager = NULL,  EntityStorageInterface $user_storage) {
    parent::__construct($config_factory,$typedConfigManager);
    $this->userStorage = $user_storage;
  }

  /**
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')->getStorage('user')
    );
  }


  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'mapping_service_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['mapping_service.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['mapping_user'] = [
      '#type' => 'select',
      '#title' => $this->t('Content creator'),
      '#description' => $this->t('The user that is assigned the author for the newly created nodes.'),
      '#options' => $this->getUsers(),
      '#default_value' =>  $this->config('mapping_service.settings')->get('mapping_user')?? 1
    ];
    return parent::buildForm($form, $form_state);
  }


  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('mapping_service.settings')
      ->set('mapping_user', $form_state->getValue('mapping_user'))
      ->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * @return array<User>
   *
   * TODO Limit by role or create <entity> rights.
   */
  private function getUsers(){
      /** @var User $users */
      $users = $this->userStorage->loadByProperties([]);
      $result = [];
      foreach ($users as $user) {
        $result[$user->id()] = $user->getDisplayName();
      }
      return $result;
    }
}
