<?php

namespace Drupal\mapping_service\Concerns;


use Drupal\mapping_service\Attribute\GetValue;
use Drupal\mapping_service\Attribute\SetValue;
use Drupal\mapping_service\Contract\HasParentInterface;
use Drupal\services_base\Str;
use ReflectionException;
use ReflectionProperty;

trait HasAttributes {

  protected array $attributes = [];

  protected array $annotatedSetAttributes = [];

  protected array $annotatedGetAttributes = [];

  public static function fromValues($values): static {
    $object = new static();
    $object->fill($values);
    return $object;
  }

  public function fill($values): void {
    foreach ($values as $key => $value) {
      $this->setAttribute($key, $value);
    }
  }

  public function setAttribute($key, $value) {
    if (method_exists($this, $methodName = 'set' . Str::studly($key))) {
      return $this->attributes[$key] = $this->{$methodName}($value);
    }
    if (!$this->setAnnotationMutator($key, $value)) {
      return $this->attributes[$key] = $value;
    }
  }




  /**
   * Little helper to decorate your class that has magic attributes :)
   * TODO: Move to 'debugger trait'
   */
  public function getAttributesAsProperties(): void {
    echo "\n" . implode("\n", array_map(static fn($key) => "private mixed \$$key;\n", array_keys($this->attributes))) . "\n";
    exit();
  }

  public function __get($key) {
    return $this->getAttribute($key);
  }

  public function __set($key, $value) {
    $this->setAttribute($key, $value);
  }

  protected function getAttribute($key) {
    if ($this->hasAttribute($key)) {
      if (method_exists($this, $methodName = 'get' . Str::studly($key))) {
        return $this->{$methodName}();
      }

      $value = $this->getAnnotationMutator($key);
      if ($value) {
        return $value;
      }
      return $this->attributes[$key];
    }
  }

  public function hasAttribute($key): bool {
    return (array_key_exists($key, $this->attributes));
  }

  /**
   * @throws \ReflectionException
   */
  protected function getAnnotationMutator($key) {
    if (isset($this->annotatedGetAttributes[$key])) {
      $getter = $this->annotatedGetAttributes[$key];
    }
    else {
      $getter = false;
      try {
        $reflectionProperty = new ReflectionProperty($this, $key);
        $getAttributes = $reflectionProperty->getAttributes(GetValue::class);

        if (!empty($getAttributes)) {
          $getter = $getAttributes[0]->newInstance();
          $this->annotatedGetAttributes[$key] = $getter;
        }
      } catch (ReflectionException $e) {
        return NULL;
      }
    }
    if ($getter) {
      return $getter->get($this->{$key});
    }
    return NULL;
  }

  protected function setAnnotationMutator($key, $value): bool {
    if (isset($this->annotatedSetAttributes[$key])) {
      $setter = $this->annotatedSetAttributes[$key];
    }
    else {
      $setter = FALSE;
      try {
        $reflectionProperty = new ReflectionProperty($this, $key);
        $setAttributes = $reflectionProperty->getAttributes(SetValue::class);
        if (!empty($setAttributes)) {
          $setter = reset($setAttributes)->newInstance()->getInstance();
          if ($setter instanceof HasParentInterface) {
            $setter->setParent($this);
          }
          $this->annotatedSetAttributes[$key] = $setter;
        }
      } catch (ReflectionException $e) {
        return FALSE;
      }
    }
    if ($setter) {
      $this->attributes[$key] = $setter->set($value);
      return TRUE;
    }
    return FALSE;
  }


  // RB Not sure about this one, it should check the $attributes array?

  public function __isset($key) {
    return $this->isset($key);
  }

  /**
   * @throws \ReflectionException
   */
  public function isset($key): bool {
    return (new ReflectionProperty(self::class, $key))->isInitialized($this);
  }
}
