<?php

namespace Drupal\lazy_responsive_image\Plugin\Field\FieldFormatter;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Template\Attribute;
use Drupal\Core\Url;
use Drupal\file\FileInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\image\ImageStyleInterface;
use Drupal\image\ImageStyleStorageInterface;
use Drupal\image\Plugin\Field\FieldFormatter\ImageFormatterBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'image' formatter.
 *
 * @FieldFormatter(
 *   id = "lazy_responsive_images",
 *   label = @Translation("Lazy responsive images"),
 *   field_types = {
 *     "image"
 *   },
 *   quickedit = {
 *     "editor" = "image"
 *   }
 * )
 */
class LazyImageFieldFormatter extends ImageFormatterBase {


  protected AccountInterface $currentUser;

  protected ImageStyleStorageInterface $imageStyleStorage;

  protected ImageStyleInterface $base64BaseImageStyle;


  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, AccountInterface $current_user, EntityStorageInterface $image_style_storage) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);
    $this->currentUser = $current_user;
    $this->imageStyleStorage = $image_style_storage;
    $this->base64BaseImageStyle = $image_style_storage->load('lazy_responsive_placeholder');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('current_user'),
      $container->get('entity_type.manager')->getStorage('image_style')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'image_style' => '',
        'lazy_loaded' => TRUE,
        'sizes' => '',
        'base64_placeholder' => FALSE,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $image_styles = $this->getSelectableImageStyles();

    $description_link = Link::fromTextAndUrl(
      $this->t('Configure Responsive Image Styles'),
      Url::fromRoute('lazy_responsive_image.generate')
    );
    $element['image_style'] = [
      '#title' => t('Responsive Image style'),
      '#type' => 'select',
      '#default_value' => $this->getSetting('image_style'),
      '#options' => $image_styles,
      '#description' => $description_link->toRenderable() + [
          '#access' => $this->currentUser->hasPermission('administer image styles'),
        ],
    ];
    $element['base64_placeholder'] = [
      '#title' => $this->t('Use a small Base64 encoded placeholder'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('base64_placeholder'),
    ];
    $element['lazy_loaded'] = [
      '#title' => $this->t('Lazy load the images'),
      '#type' => 'checkbox',
      '#default_value' => $this->getSetting('lazy_loaded'),
    ];

    $element['sizes'] = [
      '#title' => $this->t('Sizes rule'),
      '#type' => 'textfield',
      '#default_value' => $this->getSetting('sizes'),
      '#description' => $this->t('Optional sizes rule for example: <em>(min-width: 720px) 500px, 400px</em>'),
    ];


    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $image_style_setting = $this->getSetting('image_style');
    $summary[] = $this->t('Responsive image style: @responsive_image_style', ['@responsive_image_style' => $image_style_setting]);
    if (!empty($this->getSetting('sizes'))) {
      $summary[] = $this->t('Sizes: @sizes', ['@sizes' => $this->getSetting('sizes')]);
    }
    if ((bool) $this->getSetting('lazy_loaded')) {
      $summary[] = $this->t('Lazy loaded');
    }
    if ((bool) $this->getSetting('base64_placeholder')) {
      $summary[] = $this->t('With Base64 placeholder');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $files = $this->getEntitiesToView($items, $langcode);

    // Early opt-out if the field is empty.
    if (empty($files)) {
      return $elements;
    }

    $base_cache_tags = [];
    $image_styles = $this->getImageStyles();
    // Generate the cache tags for the image styles, for now assume there is only 1 per image style
    if (!empty($image_styles)) {
      $base_cache_tags = array_keys($image_styles);
      array_walk($base_cache_tags, static function (&$imageStyle) {
        $imageStyle = 'config:image.style.' . $imageStyle;
      });
    }
    $baseStyleID = $this->getBaseImageStyleID();
    $lazy_loaded = (bool) $this->getSetting('lazy_loaded');
    $base64_placeholder = (bool) $this->getSetting('base64_placeholder');

    foreach ($files as $delta => $file) {
      $cache_tags = Cache::mergeTags($base_cache_tags, $file->getCacheTags());
      /** @var  $file FileInterface */
      $image_uri = $file->getFileUri();
      $image_style_uri = $this->generateImageStyleURI($image_styles, $image_uri, $baseStyleID);
      $image_srcset = $this->generateSRCSet($image_style_uri, $file);
      if ($base64_placeholder) {
        $src_image = $this->generateBase64Image($image_uri);
      }
      else {
        $src_image = $this->base64BaseImageStyle->buildUrl($image_uri);
      }

      $elements[$delta] = [
        '#theme' => 'lazy_responsive_image',
        '#file' => $file,
        '#image_style_uri' => $image_style_uri,
        '#responsive_image_style_id' => $this->getBaseImageStyleID(),
        '#lazy_loaded' => $this->getSetting('lazy_loaded'),
        '#sizes' => $this->getSetting('sizes'),
        '#srcset' => $image_srcset,
        '#src' => $src_image,
        '#cache' => [
          'tags' => $cache_tags,
        ],
      ];
      if ($lazy_loaded) {
        $elements[$delta]['#attributes'] = new Attribute(['class' => ['lazyload']]);
        $elements[$delta]['#attached'] = [
          'library' => [
            'lazy_responsive_image/sizes',
            'lazy_responsive_image/resizer',
          ],
        ];
      }
    }
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    $dependencies = parent::calculateDependencies();
    $style_id = $this->getSetting('image_style');
    /** @var \Drupal\image\ImageStyleInterface $style */
    if ($style_id && $style = ImageStyle::load($style_id)) {
      // If this formatter uses a valid image style to display the image, add
      // the image style configuration entity as dependency of this formatter.
      $dependencies[$style->getConfigDependencyKey()][] = $style->getConfigDependencyName();
    }
    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function onDependencyRemoval(array $dependencies) {
    $changed = parent::onDependencyRemoval($dependencies);
    $style_id = $this->getSetting('image_style');
    /** @var \Drupal\image\ImageStyleInterface $style */
    if ($style_id && $style = ImageStyle::load($style_id)) {
      if (!empty($dependencies[$style->getConfigDependencyKey()][$style->getConfigDependencyName()])) {
        $replacement_id = $this->imageStyleStorage->getReplacementId($style_id);
        if ($replacement_id && ImageStyle::load($replacement_id)) {
          $this->setSetting('image_style', $replacement_id);
          $changed = TRUE;
        }
      }
    }
    return $changed;
  }

  /**
   * Build the dropdown of selectable image styles
   */
  protected function getSelectableImageStyles(): array {
    $aspects = array_filter(preg_split('/\s+/', \Drupal::config('lazy_responsive_image.settings')
      ->get('aspect_ratios')));
    $availableStyles = [];
    foreach ($aspects as $aspect) {
      $availableStyles[$aspect] = $aspect;
    }
    $availableStyles['width'] = $this->t('Fixed width');
    if (!empty(\Drupal::config('lazy_responsive_image.settings')
      ->get('minimum_height'))) {
      $availableStyles['height'] = $this->t('Fixed height');
    }
    return $availableStyles;
  }

  /**
   * Get all the available imageStyles for the given configuration.
   */
  protected function getImageStyles(): array {

    $type = $this->getBaseImageStyleID();
    $selectedIDs = $this->imageStyleStorage->getQuery()
      ->condition('name', $type . '_', 'STARTS_WITH')
      ->execute();
    return $this->imageStyleStorage->loadMultiple($selectedIDs);
  }

  /**
   * get the prefix for the baseImageStyle
   */
  private function getBaseImageStyleID(): string {
    $type = $this->getSetting('image_style');
    if ($type !== 'width') {
      $type = str_replace(':', '_', $type);
    }
    return 'responsive_' . $type;
  }


  /**
   * Generate all the URI for the given image style
   */
  protected function generateImageStyleURI(array $image_styles, string $image_uri, string $baseStyleID): array {
    $image_style_uri = [];
    $webp_available = (\Drupal::moduleHandler()
        ->moduleExists('webp') && isset($_SERVER['HTTP_ACCEPT']) && strpos($_SERVER['HTTP_ACCEPT'], 'image/webp') !== FALSE);
    $fileUrlGenerator = \Drupal::service('file_url_generator');
    foreach ($image_styles as $image_style) {
      if ($image_uri || $image_style || $image_style->supportsUri($image_uri)) {
        $imageStyleID = $image_style->getName();
        $imageStyleSize = str_ireplace($baseStyleID . '_', '', $imageStyleID);
        $path_parts = pathinfo($image_uri);
        if ($path_parts['extension'] === 'gif') {
          $imageStyleURI = $fileUrlGenerator->transformRelative($image_uri);
        }
        else {
          $imageStyleURI = $fileUrlGenerator->transformRelative($image_style->buildUrl($image_uri));
          if ($webp_available) {
            $original_extension = '.' . $path_parts['extension'];
            if (FALSE !== $pos = strrpos($imageStyleURI, $original_extension)) {
              $imageStyleURI = substr_replace($imageStyleURI, '.webp', $pos, strlen($original_extension));
            }
          }
        }
        $image_style_uri[$imageStyleSize] = $imageStyleURI;
      }
    }
    ksort($image_style_uri, SORT_NUMERIC);
    return $image_style_uri;
  }

  /**
   * Concat the image URI into a srcset string
   */
  protected function generateSRCSet(array $image_style_uri, $file): string {
    $image_srcset = [];
    foreach ($image_style_uri as $size => $uri) {
      $image_srcset[] = sprintf('%s %s', $uri, $size);
    }
    return (implode(',', $image_srcset));
  }

  public function generateBase64Image(string $sourceURI): string {

    $tmpImageUri = $this->base64BaseImageStyle->buildUri($sourceURI);
    if (!file_exists($tmpImageUri)) {
      $image_derivative = $this->base64BaseImageStyle->createDerivative($sourceURI, $tmpImageUri);
      // If the image derivative could not be generated return a 1x1 white pixel
      if (!$image_derivative) {
        return 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAQAAAC1HAwCAAAAC0lEQVR42mNkYAAAAAYAAjCB0C8AAAAASUVORK5CYII=';
      }
    }
    $mimeTypeGuesser = \Drupal::getContainer()->get('file.mime_type.guesser');
    $mimeType = $mimeTypeGuesser->guessMimeType($tmpImageUri);
    $base_64_image = base64_encode(file_get_contents($tmpImageUri));
    return "data:{$mimeType};base64,{$base_64_image}";
  }
}
