/**
 * @file
 * Attach the ResizeObserver to responsive images to load the best image style.
 */
(function (Drupal) {

  'use strict';

  Drupal.behaviors.lazyResizer = {
    observer: {},
    images: [],

    detach: function (context){
      // Detach observed images
      this.images.forEach(image => {
        this.observer.unobserve(image.parentNode);
      });

    },
    attach: function (context) {
      // Fetch all images containing a "data-srcset" attribute.
      this.images = context.querySelectorAll('img[data-srcset]');
      // Create a ResizeObserver to update the image "src" attribute when its
      // parent container resizes.
      this.observer = new ResizeObserver(entries => {
          for (let entry of entries) {
          // Observe the entries for this entry #inception
          const entryImages = entry.target.querySelectorAll('img[data-srcset]');
          entryImages.forEach(entryImage => {
            const availableWidth = Math.floor(entry.target.clientWidth);
            const attrWidth = entryImage.getAttribute('width');
            const sources = entryImage.getAttribute('data-srcset').split(',');
            // no upscaling!
            if (attrWidth && attrWidth > availableWidth) {
              return;
            }

            // Find the best matching source for the image space.
            let source, responsiveImgPath, responsiveImgWidth;
            for (source of sources) {
              let array = source.split(' ');
              responsiveImgPath = array[0];
              responsiveImgWidth = array[1].slice(0, -1);
              if (availableWidth < responsiveImgWidth) {
                break;
              }
            }

            // Update the "src" with the new image and also set the "width"
            // attribute to easily check if we need a new image after resize.
            entryImage.setAttribute('src', responsiveImgPath);
            entryImage.setAttribute('width', responsiveImgWidth);
          });

        }
      });
      // Attach the ResizeObserver to the image containers.
      this.images.forEach(image => {
        this.observer.observe(image.parentNode);
      });

    }
  };

})(Drupal);
