<?php

namespace Drupal\lazy_responsive_image\Form\Config;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\image\ImageStyleStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;


class GenerateImageStyles extends ConfigFormBase {

  protected EntityTypeManagerInterface $entityTypeManager;


  protected ModuleHandlerInterface $moduleHandler;

  protected ImageStyleStorageInterface $imageStyleStorage;

  protected array $generatedStyles = [];

  protected bool $useFocalPoint = FALSE;

  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler) {
    parent::__construct($config_factory);
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $module_handler;
    $this->imageStyleStorage = $this->entityTypeManager->getStorage('image_style');
  }


  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('module_handler')
    );
  }


  public function getFormId() {
    return 'responsive_images_generator';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['lazy_responsive_image.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Global settings'),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    $form['settings']['cleanup'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Remove old generated image styles'),
      '#description' => $this->t('Remove all the image styles that start with `responsive_` that are not generated in this run.'),
      '#default_value' => FALSE,
    ];

    if ($this->moduleHandler->moduleExists('focal_point')) {
      $form['settings']['focal_point'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Use Focal point image style'),
        '#default_value' => TRUE,
      ];
    }

    $form['aspect'] = [
      '#type' => 'details',
      '#title' => $this->t('Width'),
      '#description' => $this->t("Image styles will be generated between the minimum and maximum width with for images with a specific width but flexible height, maintaining the original aspect ratio of the image. To optimize the number of generated styles a fixed difference between the image style width can be configured. If there is a need to generate cropped images with a specific aspect ratio, a list of aspect ratio's can be defined."),
      '#open' => TRUE,
      '#tree' => FALSE,
    ];

    $form['aspect']['threshold_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Preferred amount of pixels between the width of each image style'),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('threshold_width'),
      '#min' => 10,
      '#max' => 500,
      '#step' => 10,
      '#field_suffix' => ' ' . $this->t('pixels'),
    ];

    $form['aspect']['minimum_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum image style width'),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('minimum_width'),
      '#min' => 50,
      '#max' => 1000,
      '#step' => 50,
      '#field_suffix' => ' ' . $this->t('pixels'),
    ];

    $form['aspect']['maximum_width'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum image style width'),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('maximum_width'),
      '#min' => 50,
      '#max' => 3000,
      '#step' => 50,
      '#field_suffix' => ' ' . $this->t('pixels'),
    ];

    $form['aspect']['aspect_ratios'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Supported aspect ratios'),
      '#description' => $this->t("Define a list of aspect ratio's in the format w:h (eg. 16:9 or 4:3). Place each aspect ration on a separate line."),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('aspect_ratios'),
      '#attributes' => ['placeholder' => $this->t('eg. 16:9')],
    ];

    $form['height'] = [
      '#type' => 'details',
      '#title' => $this->t('Height'),
      '#description' => $this->t('Define a minimum and maximum height if there is a need for images that have a specific height but a flexible width. This will not influence the image styles generated based on aspect ratio.'),
      '#tree' => FALSE,
      '#open' => TRUE,
    ];

    $form['height']['threshold_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Preferred amount of pixels between the height of each image style'),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('threshold_height'),
      '#min' => 10,
      '#max' => 500,
      '#step' => 10,
      '#field_suffix' => ' ' . $this->t('pixels'),
    ];

    $form['height']['minimum_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum image style height'),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('minimum_height'),
      '#min' => 50,
      '#max' => 1000,
      '#step' => 50,
      '#field_suffix' => ' ' . $this->t('pixels'),
    ];

    $form['height']['maximum_height'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum image style height'),
      '#default_value' => $this->config('lazy_responsive_image.settings')
        ->get('maximum_height'),
      '#min' => 100,
      '#max' => 3000,
      '#step' => 50,
      '#field_suffix' => ' ' . $this->t('pixels'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('lazy_responsive_image.settings');

    if ((bool) $form_state->getValue('focal_point')) {
      $this->useFocalPoint = TRUE;
    }

    $width_threshold = $form_state->getValue('threshold_width') ?? 100;
    $width_min = $form_state->getValue('minimum_width');
    $width_max = $form_state->getValue('maximum_width');

    $height_threshold = $form_state->getValue('threshold_height') ?? 100;
    $height_min = $form_state->getValue('minimum_height');
    $height_max = $form_state->getValue('maximum_height');


    $aspect_ratios = $form_state->getValue('aspect_ratios');

    $config->set('threshold_width', $width_threshold)
      ->set('minimum_width', $width_min)
      ->set('maximum_width', $width_max)
      ->set('aspect_ratios', $aspect_ratios)
      ->set('threshold_height', $height_threshold)
      ->set('minimum_height', $height_min)
      ->set('maximum_height', $height_max)
      ->set('use_focal_point', $this->useFocalPoint)
      ->save();

    // Not sure if this is config or not..



    $this->generateFixedWidthFlexibleHeightStyles($width_min, $width_max, $width_threshold);
    $this->generateAspectRatioImageStyles($aspect_ratios, $width_min, $width_max, $width_threshold);
    $this->generateFixedHeightFlexibleWidthStyles($height_min, $height_max, $height_threshold);

    $obsolete_styles = [];
    if ((bool) $form_state->getValue('cleanup')) {
      // Ask extra confirmation?

      $obsolete_ids = $this->imageStyleStorage->getQuery()
        ->condition('name', 'responsive_', 'STARTS_WITH')
        ->condition('name', $this->generatedStyles, 'NOT IN')
        ->execute();
      $obsolete_styles = $this->imageStyleStorage->loadMultiple($obsolete_ids);
      $this->imageStyleStorage->delete($obsolete_styles);
    }

    $this->messenger()
      ->addStatus($this->t('Created %new styles, deleted %old styles',
        [
          '%new' => count($this->generatedStyles),
          '%old' => count($obsolete_styles),
        ]));
  }


  /**
   * Generate the image styles for images with a specific width but flexible
   * height, maintaining the original aspect ratio of the image.
   */
  protected function generateFixedWidthFlexibleHeightStyles(int $width_min, int $width_max, int $width_threshold): void {
    if ($width_min && $width_max) {
      for ($width = $width_min; $width <= $width_max; $width += $width_threshold) {
        $name = 'responsive_width_' . $width . 'w';
        $generated_styles[] = $name;
        $effect = [
          'id' => 'image_scale',
          'data' => [
            'width' => $width,
            'height' => NULL,
            'upscale' => TRUE,
          ],
        ];
        $this->createImageStyle($name, $effect);
      }
    }
  }

  /**
   * Generate the image styles using the desired aspect ratios
   * With focal_point enabled if applicable.
   */
  protected function generateAspectRatioImageStyles($aspect_ratios, int $width_min, int $width_max, int $width_threshold): void {

    $aspect_ratios = array_map(static function ($aspect_ratio){
      return [$w, $h] = explode(':', $aspect_ratio);
    },array_filter(preg_split('/\s+/',$aspect_ratios)));


    foreach ($aspect_ratios as $aspect_ratio) {
      [$w, $h] = $aspect_ratio;

      for ($width = $width_min; $width <= $width_max; $width += $width_threshold) {
        $height = ($width / $w) * $h;
        $name = 'responsive_' . $w . '_' . $h . '_' . $width . 'w';
        $generated_styles[] = $name;

        $effect = [
          'id' => 'image_scale_and_crop',
          'data' => [
            'width' => $width,
            'height' => $height,
          ],
        ];
        if ($this->useFocalPoint) {
          $effect['id'] = 'focal_point_scale_and_crop';
          $effect['data']['crop_type'] = 'focal_point';
        }
        $this->createImageStyle($name, $effect);
      }
    }
  }

  /**
   * Generate the image styles for images with a specific height but flexible
   * width, maintaining the original aspect ratio of the image.
   */
  protected function generateFixedHeightFlexibleWidthStyles($height_min, $height_max, $height_threshold): void {
    if ($height_min && $height_max) {
      for ($height = $height_min; $height <= $height_max; $height += $height_threshold) {
        $name = 'responsive_height_' . $height . 'h';
        $effect = [
          'id' => 'image_scale',
          'data' => [
            'width' => NULL,
            'height' => $height,
            'upscale' => TRUE,
          ],
        ];
        $this->createImageStyle($name, $effect);
      }
    }
  }

  /**
   * Generate and save a image style with the given effect
   */
  protected function createImageStyle(string $name, array $effect): void {
    $this->generatedStyles[] = $name;
    if (!$this->imageStyleStorage->load($name)) {
      $style = $this->imageStyleStorage->create([
        'name' => $name,
        'label' => $name,
      ]);
      $style->addImageEffect($effect);
      $style->save();

    }
  }

}
