<?php

namespace Drupal\graphql_layoutbuilder\Plugin\GraphQL\Fields\Components;

use Drupal\block_content\Plugin\Block\BlockContentBlock;
use Drupal\Core\Block\BlockBase;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\entity_browser_block\Plugin\Block\EntityBrowserBlock;
use Drupal\graphql\GraphQL\Cache\CacheableValue;
use Drupal\graphql\GraphQL\Execution\ResolveContext;
use Drupal\graphql\Plugin\GraphQL\Fields\FieldPluginBase;
use Drupal\graphql\Plugin\TypePluginManagerInterface;
use Drupal\layout_builder\Plugin\Block\FieldBlock;
use Drupal\layout_builder\Plugin\Block\InlineBlock;
use Drupal\layout_builder\SectionComponent;
use GraphQL\Type\Definition\ResolveInfo;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * List all blocks of a component.
 *
 * @GraphQLField(
 *   id = "component_blocks",
 *   secure = true,
 *   name = "content",
 *   type = "Entity",
 *   parents = {"Component"},
 * )
 */
class ComponentBlocks extends FieldPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * @var \Drupal\graphql\Plugin\TypePluginManagerInterface
   */
  protected $typeManager;

  /**
   * The entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface
   */
  protected $entityRepository;

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $pluginId,
    $pluginDefinition
  ) {
    return new static(
      $configuration,
      $pluginId,
      $pluginDefinition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.graphql.type'),
      $container->get('entity.repository')

    );
  }

  /**
   * BlocksByRegion constructor.
   *
   * @param array $configuration
   *   The plugin configuration array.
   * @param string $pluginId
   *   The plugin id.
   * @param mixed $pluginDefinition
   *   The plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    array $configuration,
    $pluginId,
    $pluginDefinition,
    EntityTypeManagerInterface $entityTypeManager,
    TypePluginManagerInterface $manager,
    EntityRepositoryInterface $entityRepository
  ) {
    parent::__construct($configuration, $pluginId, $pluginDefinition);
    $this->entityTypeManager = $entityTypeManager;
    $this->typeManager = $manager;
    $this->entityRepository = $entityRepository;
  }

  /**
   * {@inheritdoc}
   */
  protected function resolveValues($value, array $args, ResolveContext $context, ResolveInfo $info) {
    if ($value instanceof SectionComponent) {

      $plugin = $value->getPlugin();

      // Set the block plugin as a context.
      $context->setContext('block', $plugin, $info);

      // Reusable block from block library.
      if ($plugin instanceof BlockContentBlock) {
        $block = $this->entityRepository->loadEntityByUuid('block_content', $plugin->getDerivativeId());
        yield new CacheableValue($block, [$value]);
      }

      // LayoutBuilder inline block (not reusable).
      if ($plugin instanceof InlineBlock) {
        $configuration = $plugin->getConfiguration();
        if ($entity_rev_id = $configuration['block_revision_id']) {
          if (($entity = $this->entityTypeManager->getStorage('block_content')->loadRevision(
              $entity_rev_id
            )) && $entity->access('view')) {
            yield new CacheableValue($entity, [$value]);
          }
        }
      }

      // Entity provided by entity_block module.
      if ($plugin instanceof EntityBrowserBlock) {
        // Set entities as a context.
        $entity = NULL;
        $configuration = $plugin->getConfiguration();
        if ($entity_ids = $configuration['entity_ids']) {
          // EntityBrowser is configured to allow only one entity.
          $entities = $this->loadEntities($entity_ids);
          $entity = reset($entities);
        }
        $context->setContext('entity_browser_entity', $entity, $info);

        yield new CacheableValue($plugin, [$value]);
      }

      // FieldBlock by Layoutbuilder.
      // It is too complicated to expose a single field. For now, just expose the parent node.
      if ($plugin instanceof FieldBlock) {
        $entity = $context->getContext('entity', $info, NULL);
        if ($entity) {
          yield new CacheableValue($entity, [$value]);
        }
      }

      if ($plugin instanceof BlockBase) {
        yield new CacheableValue($plugin, [$value]);
      }
    }
  }

  /**
   * Loads entities based on an ID in the format entity_type:entity_id.
   *
   * @param array $ids
   *   An array of IDs.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   An array of loaded entities, keyed by an ID.
   */
  public function loadEntities($ids) {
    $storages = [];
    $entities = [];
    foreach ($ids as $id) {
      list($entity_type_id, $entity_id) = explode(':', $id);
      if (!isset($storages[$entity_type_id])) {
        $storages[$entity_type_id] = $this->entityTypeManager->getStorage($entity_type_id);
      }
      $entities[$entity_type_id . ':' . $entity_id] = $storages[$entity_type_id]->load($entity_id);
    }
    return $entities;
  }

}
