<?php

namespace Drupal\graphql\Commands;

use Drupal\Component\Serialization\Json;
use Drupal\graphql\GraphQL\Utility\Introspection;
use Drush\Commands\DrushCommands;
use Drush\Drush;
use Symfony\Component\Filesystem\Filesystem;
use Webmozart\PathUtil\Path;


class GraphqlCommands extends DrushCommands {

  public function __construct(Introspection $introspectionService) {
    $this->introspectionService = $introspectionService;
    $this->filesystem = new Filesystem();
  }


  protected Introspection $introspectionService;

  protected Filesystem $filesystem;

  /**
   *
   * @param string $schema Schema to export (default:default)
   * @param string $filename filename of the export (drupal_schema.json)
   * @param string $destination Path to a directory for writing the schema to
   *
   * @usage graphql:export schema destination
   *
   * @command graphql:export
   * @bootstrap max
   * @aliases gql-export
   * @validate-module-enabled dblog
   */
  public function exportCommand(string $schema = '', string $filename = '', string $destination = '') {
    if (!$this->input->getArgument('schema')) {
      $schema = $this->io()->ask('schema to export', 'default:default');
      $this->input->setArgument('schema', $schema);
    }
    if (!$this->input->getArgument('filename')) {
      $filename = $this->io()
        ->ask('Path filename of the export', 'drupal_schema.json');
      $this->input->setArgument('filename', $filename);
    }

    if (!$this->input->getArgument('destination')) {
      if ($composerRoot = Drush::bootstrapManager()->getComposerRoot()) {
        $default = Path::join($composerRoot, 'graphql');
      }
      else {
        $default = Path::join(DRUPAL_ROOT, '/graphql/drupal_schema.json');
      }
      $destination = $this->io()
        ->ask('Absolute path to a directory for writing ', $default);
      $this->input->setArgument('destination', $destination);
    }


    $this->filesystem->mkdir($destination);
    $this->writeGraphqlScheme($schema, Path::join($destination, $filename));
  }


  public function writeGraphqlScheme(string $schema, string $destination): void {

    $introspectionResult = $this->introspectionService->introspect($schema);
    $this->sortSchema($introspectionResult['data']['__schema']);
    $introspectionResultJson = Json::encode($introspectionResult);
    $this->filesystem->dumpFile($destination, $introspectionResultJson);
    $this->logger()
      ->info('The schema has been exported the GraphQL schema to ' . $destination);
  }

  private function sortSchema(&$schema): void {
    if (!empty($schema['types'])) {
      usort($schema['types'], [$this, 'compareName']);

      foreach ($schema['types'] as &$type) {
        if (!empty($type['fields'])) {
          usort($type['fields'], [$this, 'compareName']);
          foreach ($type['fields'] as &$field) {
            if (!empty($field['args'])) {
              usort($field['args'], [$this, 'compareName']);
            }
          }
        }
        if (!empty($type['inputFields'])) {
          usort($type['inputFields'], [$this, 'compareName']);
        }
        if (!empty($type['interfaces'])) {
          usort($type['interfaces'], [$this, 'compareName']);
        }
        if (!empty($type['enumValues'])) {
          usort($type['enumValues'], [$this, 'compareName']);
        }
        if (!empty($type['possibleTypes'])) {
          usort($type['possibleTypes'], [$this, 'compareName']);
        }
      }
    }

    if (!empty($schema['directives'])) {
      usort($schema['directives'], [$this, 'compareName']);
      foreach ($schema['directives'] as &$directive) {
        if (!empty($directive['args'])) {
          usort($directive['args'], [$this, 'compareName']);
        }
      }
    }
  }

  private function compareName($a, $b): int {
    return strcmp($a['name'], $b['name']);
  }

}