<?php

namespace Drupal\ga4_server\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\ga4_server\Enum\EventStatus;

class StatusController extends ControllerBase {

  /**
   * Displays the GA4 queue status overview.
   *
   * @return array
   *   A render array for the status page.
   */
  public function __invoke() {
    // Get counts for each status
    $counts = $this->getStatusCounts();
    
    // Create a container for our status blocks
    $build = [
      '#type' => 'container',
      '#attributes' => ['class' => ['ga4-status-counters']],
      '#attached' => [
        'library' => ['ga4_server/status_page'],
      ],
    ];
    
    // Add a block for each status
    foreach (EventStatus::cases() as $status) {
      $count = $counts[$status->value] ?? 0;
      $class = $this->getStatusClass($status);
      
      $build[$status->name] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['ga4-status-counter', $class]],
        'count' => [
          '#markup' => '<span class="counter-number">' . $count . '</span>',
        ],
        'label' => [
          '#markup' => '<span class="counter-label">' . $status->label() . '</span>',
        ],
      ];
    }
    
    return $build;
  }

  /**
   * Gets counts of events for each status from the database.
   *
   * @return array
   *   An array of counts keyed by status value.
   */
  protected function getStatusCounts() {
    return \Drupal::service('ga4_server.events_storage')->getEventCountsByStatus();
  }

  /**
   * Gets a CSS class for a status.
   */
  protected function getStatusClass(EventStatus $status) {
    return match($status) {
      EventStatus::NEW => 'status-new',
      EventStatus::QUEUED => 'status-warning',
      EventStatus::PROCESSED => 'status-ok',
    };
  }
}