<?php

namespace Drupal\ga4_server\GAEvent;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Component\Utility\Xss;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ga4_server\Abstract\GaEventPlugin;
use Drupal\ga4_server\Attribute\GaEvent;

#[GaEvent(
  id: 'link-click',
  label: new TranslatableMarkup('Clicked Link'),
  description: new TranslatableMarkup('Register an clicked link.'),
)]
class LinkClicked extends GaEventPlugin {


  public function getEventType(): string {
    return 'click';
  }

  public function setEventData($eventData): void {
    if ($this->hasClientID()){
      $this->validateAndSanitizeEventData((array) $eventData);
      $this->eventData['outbound'] = $this->isExternalUrl($this->eventData['link_url']);
      $this->storeEvent();
    }
  }

  private function validateAndSanitizeEventData(array $eventData): void {
    $allowedKeys = ['link_url', 'page_location', 'link_text'];
    foreach ($allowedKeys as $key) {
      if (isset($eventData[$key])) {
        switch ($key) {
          case 'link_url':
          case 'page_location':
            // Use Drupal's URL handling
            if (UrlHelper::isValid($eventData[$key], TRUE)) {
              // External URL - already validated
              $this->eventData[$key] = $eventData[$key];
            }
            elseif (UrlHelper::isValid($eventData[$key], FALSE)) {
              // Internal URL - already validated
              $this->eventData[$key] = $eventData[$key];
            }
            // If URL validation fails, skip this value
            break;

          case 'link_text':
            // Use Drupal's XSS filtering
            $this->eventData[$key] = Xss::filter($eventData[$key]);
            break;
        }
      }
    }
  }

}