<?php

declare(strict_types=1);

namespace Drupal\ga4_server\Abstract;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\ga4_server\Contract\GaEventInterface;
use Drupal\ga4_server\Database\EventsStorage;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Uid\Ulid;

/**
 * Base class for ga_event plugins.
 */
abstract class GaEventPlugin extends PluginBase implements GaEventInterface {

  protected array $eventData;

  /**
   * Constructs a RegisterEvent object.
   */
  public function __construct(
    $configuration, $plugin_id, $plugin_definition,
    protected readonly LoggerChannelInterface $logger,
    protected readonly Session $session,
    protected readonly EventsStorage $eventStorage,
    protected readonly LanguageManagerInterface $languageManager
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  public function isExternalUrl($url, $currentUrl = FALSE): bool {
    if (UrlHelper::isExternal($url)) {
      if (!$currentUrl) {
        $currentHost = \Drupal::request()->getHttpHost();
      }
      else {
        $currentHost = parse_url($currentUrl, PHP_URL_HOST);
      }
      $urlHost = parse_url($url, PHP_URL_HOST);
      return $urlHost !== $currentHost;
    }

    return FALSE;
  }

  public function label(): string {
    return (string) $this->pluginDefinition['label'];
  }

  public function hasClientID(): bool {
    if ($this->session->get('ga4_client_id', FALSE)){
      return true;
    }
    return false;
  }

  public function storeEvent(): void {
    $this->eventStorage->insertEvent($this->getClientId(), $this->getEventType(), $this->getMicroTimeStamp(), $this->eventData);
  }

  protected function getClientId(): string {
    $ga4ClientId = $this->session->get('ga4_client_id', FALSE);
    if (!$ga4ClientId) {
      $ga4ClientId = Ulid::generate();
      $this->session->set('ga4_client_id', $ga4ClientId);
      $this->logger->info('Created new ClientID %client_id', ['%client_id' => $ga4ClientId]);
    }
    return $ga4ClientId;
  }

  protected function getMicroTimeStamp(): string {
    $microtime_float = microtime(TRUE);
    $timestamp_micros_float = $microtime_float * 1000000;
    return sprintf('%0.0f', $timestamp_micros_float);
  }

}
