# GA4 Server Module

## Introduction
The GA4 Server module provides server-side Google Analytics 4 integration for Drupal sites, 
offering a privacy-focused alternative for tracking critical business metrics while minimizing the collection of personally identifying information through its approach of collecting events, storing them temporarily, and sending them to Google Analytics using the Measurement Protocol API.


## Configuration Options

- **Measurement ID**: Your GA4 property identifier (G-XXXXXXXXXX format)
- **API Secret**: Secret key required for Measurement Protocol authentication
- **Event Expiration**: How long to store events locally before purging (24/48 hours, 1 week, 1 month, or custom)
- **Users per run**: Maximum number of unique clients to process in each cron run (1-10)
- **Debug logging**: Enables detailed logging of events sent to GA

## Tracked Events

The module tracks the following events by default:

- **PageView**: Records page views with UTM parameter support and page metadata
- **LinkClicked**: Tracks link clicks, identifying internal/external links and capturing link text
- **FormSubmit**: Records form submission events

## Getting Measurement ID and API Secret

1. Go to your Google Analytics 4 property
2. Navigate to Admin > Data Streams
3. Click on your web data stream or create a new one
4. The Measurement ID is displayed at the top of the page
5. Under Measurement Protocol, click Create to generate an API Secret

## Creating Custom Events

To add a new event type:

1. Create a new class in `src/GAEvent/` that extends `GaEventPlugin`
2. Add the `#[GaEvent]` attribute with id, label, and description
3. Implement the `getEventType()` method to define the GA4 event type
4. Implement `setEventData()` to process and validate your event data
5. Use `$this->storeEvent()` to save the event for processing

```php
<?php

namespace Drupal\ga4_server\GAEvent;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ga4_server\Abstract\GaEventPlugin;
use Drupal\ga4_server\Attribute\GaEvent;

#[GaEvent(
  id: 'custom-event',
  label: new TranslatableMarkup('Custom Event'),
  description: new TranslatableMarkup('Tracks a custom event.'),
)]
class CustomEvent extends GaEventPlugin {

  public function getEventType(): string {
    return 'custom_event';
  }

  public function setEventData($eventData): void {
    // Validate and sanitize your event data
    $this->eventData = $eventData;
    $this->storeEvent();
  }
}
```

## Server-Side Tracking vs. traditional client side tracking

### Key Differences

The GA4 Server module provides server-side event tracking, which operates fundamentally differently from client-side tracking solutions like gtag.js:

#### What Server-Side Tracking Cannot Collect?

- **No Browser Data**: Cannot collect browser type, version, or client-side metrics
- **No Visit Duration**: Unable to track time on page or session duration
- **No Client-Side Interactions**: Cannot automatically track scrolling depth, video engagement, or dynamic page interactions
- **No Screen Dimensions**: Unable to collect viewport size, screen resolution, or device details
- **No JavaScript-Dependent Events**: Cannot capture events that rely on client-side JavaScript execution

#### Enhanced Privacy Benefits

- **Cookieless Operation**: Functions without relying on browser cookies, making it compatible with cookie-blocking browsers and privacy tools
- **No Personal Data Collection**: Collects only anonymized server event data rather than client fingerprinting
- **Reduced User Profiling**: Limits the ability for cross-site tracking and profile building
- **GDPR/Privacy Friendly**: Inherently collects less identifiable information, potentially reducing consent requirements
- **Immune to Ad-Blockers**: Server-side events are transmitted regardless of client-side blocking technologies

#### Best Used For

- Tracking critical business events (form submissions, purchases, registrations)
- Environments with high privacy requirements or significant ad-blocker usage
- Ensuring consistent measurement across all user types
- Capturing server-generated events that aren't visible to the client


```
