<?php

namespace Drupal\ga4_server\Plugin\QueueWorker;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\ga4_server\Client\MeasurementProtocol;
use Drupal\ga4_server\Database\EventsStorage;
use Drupal\ga4_server\Enum\EventStatus;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Psr\Log\LoggerInterface;

/**
 * Processes GA4 events for sending to measurement API.
 *
 * @QueueWorker(
 *   id = "ga4_server_events_processor",
 *   title = @Translation("GA4 Server Events Processor"),
 *   cron = {"time" = 60},
 * )
 */
class EventsProcessor extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  protected MeasurementProtocol $apiClient;

  protected LoggerInterface $logger;

  protected EventsStorage $eventStorage;

  public function __construct(array $configuration, $plugin_id, $plugin_definition, MeasurementProtocol $api_client, LoggerInterface $logger, EventsStorage $eventStorage) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->apiClient = $api_client;
    $this->logger = $logger;
    $this->eventStorage = $eventStorage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): EventsProcessor {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ga4_server.measurement_protocol_client'),
      $container->get('logger.channel.ga4_server'),
      $container->get('ga4_server.events_storage')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    try {
      // Get data from the queue item
      $client_id = $data['client_id'];
      $events = $data['events'];
      $event_ids = $data['event_ids'];

      // Process all events for this client
      $result = $this->apiClient->sendEvents($client_id, $events);

      // If successful, mark the events as processed or delete them
      if ($result) {
        // You could either delete the events or mark them as processed
        // Depending on your requirements
        $this->eventStorage->setEventStatus($event_ids, EventStatus::PROCESSED);

        $this->logger->info('Successfully processed @count events for client @client_id', [
          '@count' => count($events),
          '@client_id' => $client_id,
        ]);
      }
      else {
        $this->logger->error('Failed to process events for client @client_id', [
          '@client_id' => $client_id,
        ]);
        // You might want to implement a retry mechanism here
        throw new \RuntimeException('Failed to process events, will retry later');
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error processing events: @error', [
        '@error' => $e->getMessage(),
      ]);
      // Re-throw to let the Queue API know this item should be retried
      throw $e;
    }
  }

  /**
   * Delete events that have been successfully processed.
   *
   * @param array $event_ids
   *   Array of event ULIDs to delete.
   */


}

