<?php

declare(strict_types=1);

namespace Drupal\ga4_server\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure GA4 Serverside settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return "ga4_server_settings";
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ["ga4_server.settings"];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state
  ): array {
    $form["#attached"]["library"][] = "ga4_server/interval_form_field";

    $form["ga4_credentials"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["ga4-credentials-container"],
      ],
    ];

    $form["ga4_credentials"]["measurement_id"] = [
      "#type" => "textfield",
      "#title" => $this->t("Measurement ID"),
      "#placeholder" => "G-XXXXXXXXXX",
      "#default_value" => $this->config("ga4_server.settings")->get(
        "measurement_id"
      ),
      "#required" => TRUE,
    ];

    $hasSecret = is_string(
      $this->config("ga4_server.settings")->get("api_secret")
    );
    $form["ga4_credentials"]["api_secret"] = [
      "#type" => "password",
      "#title" => $this->t("API Secret"),
      "#default_value" => $this->config("ga4_server.settings")->get(
        "api_secret"
      ),
      "#required" => !$hasSecret,
      "#attributes" => [
        "autocomplete" => "off",
      ],
    ];
    if ($hasSecret) {
      $form["ga4_credentials"]["api_secret"]["#description"] = $this->t(
        "API Secret is set, set new value to update."
      );
    }
    $form["ga4_credentials"]["api_secret_howto"] = [
      "#type" => "markup",
      "#markup" => $this->t('<small>How to get your API Secret:
  <ol>
    <li>Go to your <strong>Google Analytics 4</strong> property</li>
    <li>Navigate to <em>Admin > Data Streams</em></li>
    <li>Click on your web data stream or create a new stream</li>
    <li>Under <em>Measurement Protocol</em>, click <strong>Create</strong></li>
    <li>Copy the generated API Secret and paste it above</li>
  </ol>
  </small>'),
    ];

    $form["ga4_behaviours"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["ga4-behaviours-container"],
      ],
    ];
    $form["ga4_behaviours"]["cleanup_interval"] = [
      "#type" => "container",
      "#attributes" => [
        "class" => ["cleanup-interval-container"],
      ],
      "#tree" => TRUE,
    ];

    // The time unit selector
    $form["ga4_behaviours"]["cleanup_interval"]["unit"] = [
      "#type" => "select",
      "#title" => $this->t("Expire events after"),
      "#options" => $this->getCleanupIntervalSelectOptions(),
      "#default_value" => $this->getCleanupIntervalDefaultOption(),
      "#description" => $this->t(
        "Time to store events locally before purging them. Changing this will only apply for new events. <br />Events <strong>will be deleted</strong> regardless of them being sent to GA."
      ),
    ];

    // The custom value input field
    $form["ga4_behaviours"]["cleanup_interval"]["custom_value"] = [
      "#type" => "number",
      "#title" => $this->t("Custom expiration in hours"),
      "#min" => 1,
      "#max" => 2160, // Approx. 3 months in hours
      "#default_value" => $this->getCleanupIntervalCustomValueInHours(),
      "#states" => [
        "visible" => [
          ':input[name="cleanup_interval[unit]"]' => [
            "value" => "custom",
          ],
        ],
        "required" => [
          ':input[name="cleanup_interval[unit]"]' => [
            "value" => "custom",
          ],
        ],
      ],
      "#description" => $this->t(
        "Enter a custom interval in hours (1-2160)."
      ),
    ];

    $form["clients_per_run"] = [
      "#type" => "number",
      "#title" => $this->t("Users per run"),
      "#min" => 1,
      "#max" => 40,
      "#default_value" =>
        $this->config("ga4_server.settings")->get("clients_per_run") ??
        10,
      "#description" => $this->t(
        "The maximum number of unique clients that are enqueued per cron run."
      ),
    ];

    $form['min_events_per_client'] = [
      '#type' => 'number',
      '#title' => $this->t('Minimum events per client'),
      '#min' => 1,
      '#max' => 10,
      '#default_value' =>
        $this->config('ga4_server.settings')->get('min_events_per_client') ??
        2,
      '#description' => $this->t(
        'All clients with events less than this value will never be sent to GA <br /> Default is two events, setting it to one will send all events to GA'
      ),
    ];

    // Add JavaScript to validate and ensure proper UI behavior

    $form["debug_mode"] = [
      "#type" => "checkbox",
      "#title" => $this->t("Enable debug logging"),
      "#default_value" => $this->config("ga4_server.settings")->get(
        "debug_mode"
      ),
      "#description" => $this->t(
        "Add detailed log data on events send to GA."
      ),
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state
  ): void {
    parent::validateForm($form, $form_state);
    // TODO support \Drupal\ga4_server\Client\MeasurementProtocol::validateCredentials
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config("ga4_server.settings");
    $newSecret = $form_state->getValue("api_secret");
    $cleanupInterval = $this->getCleanupIntervalInSeconds(
      $form_state->getValue("cleanup_interval")
    );
    $config->set("cleanup_interval", $cleanupInterval);
    $config->set("min_events_per_client", $cleanupInterval);
    if (!empty($newSecret)) {
      $config->set("api_secret", $newSecret);
    }

    $config->set('measurement_id', $form_state->getValue('measurement_id'));
    $config->set('debug_mode', $form_state->getValue('debug_mode'));
    $config->set('clients_per_run', $form_state->getValue('clients_per_run'));
    $config->set('min_events_per_client', $form_state->getValue('min_events_per_client'));
    $config->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Code to build up the fancy interval Time form field
   */
  private array $cleanupTimeIntervals = [
    "preset_24h" => [
      "label" => "24 hours",
      "seconds" => 86400,
    ],
    "preset_48h" => [
      "label" => "48 hours",
      "seconds" => 172800,
    ],
    "preset_1w" => [
      "label" => "1 week",
      "seconds" => 604800,
    ],
    "preset_1m" => [
      "label" => "1 month",
      "seconds" => 2592000,
    ],
    "custom" => [
      "label" => "Custom value",
      "seconds" => NULL, // Will be calculated based on user input
    ],
  ];

  /**
   * Default is 24 hours
   */
  private function getCleanupIntervalDefaultOption(): string {
    $stored_seconds =
      $this->config("ga4_server.settings")->get("cleanup_interval") ??
      86400;

    // Check if the value matches any of our presets
    foreach ($this->cleanupTimeIntervals as $key => $data) {
      if ($key !== "custom" && $data["seconds"] === $stored_seconds) {
        return $key;
      }
    }
    return "custom";
  }

  /**
   * Default is 24 hours
   */
  private function getCleanupIntervalCustomValueInHours(): ?int {
    $stored_seconds =
      $this->config("ga4_server.settings")->get("cleanup_interval") ??
      86400;
    if (
      array_any(
        $this->cleanupTimeIntervals,
        static fn($data, $key) => $key !== "custom" &&
          $data["seconds"] === $stored_seconds
      )
    ) {
      return NULL;
    }
    // Convert seconds to hours
    return (int) ($stored_seconds / 3600);
  }

  private function getCleanupIntervalInSeconds(array $interval_data): int {
    $unit = $interval_data["unit"];
    if ($unit !== "custom" && isset($this->cleanupTimeIntervals[$unit])) {
      return $this->cleanupTimeIntervals[$unit]["seconds"];
    }
    return (int) $interval_data["custom_value"] * 3600;
  }

  private function getCleanupIntervalSelectOptions(): array {
    return array_map(function($values) {
      return $this->t($values["label"]);
    }, $this->cleanupTimeIntervals);
  }

}
