<?php

declare(strict_types=1);

namespace Drupal\ga4_server\GAEvent;

use Drupal\Component\Utility\Xss;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ga4_server\Abstract\GaEventPlugin;
use Drupal\ga4_server\Attribute\GaEvent;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Event\ViewEvent;

#[GaEvent(
  id: 'page-view',
  label: new TranslatableMarkup('Pageview'),
  description: new TranslatableMarkup('Register a page view with support for UTM_ attributes.'),
)]
class PageView extends GaEventPlugin  {

  protected Request $request;

  public function getEventType(): string {
    return 'page_view';
  }

  public function setEventData($eventData): void {
    if (!$eventData instanceof ViewEvent) {
      $this->logger->error('Invalid viewEvent provided to PageView');
      return;
    }
    $this->request = $eventData->getRequest();
    $this->setUTMParameters();
    $this->setEventDataFromNode($eventData);
    $this->setReferrerData();
    $this->eventData['language'] = $this->languageManager->getCurrentLanguage()
      ->getId();
    $this->storeEvent();
  }

  /**
   * Set a referrer if it's an external hit
   */
  private function setReferrerData(): void {
    $referrer = $this->request->headers->get('referer');
    if (!$referrer) {
      return;
    }

    $currentHost = $this->request->getHttpHost();
    $referrerHost = parse_url($referrer, PHP_URL_HOST);

    if ($referrerHost !== $currentHost) {
      $this->eventData['page_referrer'] = $referrer;
    }
  }

  private function setUTMParameters(): void {
    $query = $this->request->query; // Get query parameters

    $mapping = [
      'utm_source' => 'source',
      'utm_medium' => 'medium',
      'utm_campaign' => 'campaign',
      'utm_term' => 'term',
      'utm_content' => 'content',
      'utm_id' => 'campaign_id',
      'utm_source_platform' => 'source_platform',
      'utm_creative_format' => 'creative_format',
      'utm_marketing_tactic' => 'marketing_tactic',
    ];

    foreach ($mapping as $utmKey => $ga4Key) {
      if ($query->has($utmKey)) {
        $this->eventData[$ga4Key] =  Xss::filter($query->get($utmKey));
      }
    }
  }

  private function setEventDataFromNode($viewEvent): void {
    $controller = $viewEvent->getControllerResult();
    if (isset($controller['#node'])) {
      $node = $controller['#node'];
      $canonical = $node->toUrl('canonical', [
        'absolute' => TRUE,
        'language' => $this->languageManager->getCurrentLanguage(),
      ])->toString(FALSE);
      $this->eventData['page_location'] = $canonical;
      $this->eventData['page_title'] = $node->get('title')->getString();
    }
  }



}
