<?php

declare(strict_types=1);

namespace Drupal\ga4_server\Database;

use Drupal\Core\Database\Connection;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\ga4_server\Enum\EventStatus;
use Symfony\Component\Uid\Ulid;

/**
 * Service for managing GA4 events in the database.
 */
class EventsStorage {

  /**
   * The database table name.
   *
   * @var string
   */
  protected string $table = 'events_collector';

  public function __construct(
    protected readonly Connection $database,
    protected readonly LoggerChannelInterface $loggerChannel
  ) {}

  /**
   * Insert a new event into the database.
   *
   * @param string $client_id
   *   The client identifier.
   * @param string $event_type
   *   The type of event.
   * @param string $event_microtime
   * The microtime stamp of an event.
   * @param array $event_data
   *   The JSON-serializable event data.
   */
  public function insertEvent(string $client_id, string $event_type, string $event_microtime, array $event_data): void {
    $cleanupInterval = \Drupal::config('ga4_server.settings')->get('cleanup_interval');
    $timeOffset = "+ $cleanupInterval seconds";
    $expire = new DrupalDateTime()->modify($timeOffset)->getTimestamp();
    $ulid = Ulid::generate();
    try {
      $this->database->insert($this->table)
        ->fields([
          'ulid' => $ulid,
          'client_id' => $client_id,
          'event_type' => $event_type,
          'event_timestamp' => $event_microtime,
          'event_data' => json_encode($event_data, JSON_THROW_ON_ERROR),
          'expire' => $expire,
          'created' => time(),
          'status' => EventStatus::NEW->value
        ])
        ->execute();
    }
    catch (\Exception $e) {
      $this->loggerChannel->notice('Could not store event %event_type : %error ', [
        '%event_id' => $event_type,
        '%error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Delete an event by ID.
   *
   * @param int $event_id
   *   The event ID to delete.
   *
   * @return bool
   *   TRUE if the deletion was successful, FALSE otherwise.
   */
  public function deleteEvent(string $event_id): bool {
    try {
      $affected = $this->database->delete($this->table)
        ->condition('ulid', $event_id)
        ->execute();

      return $affected > 0;
    }
    catch (\Exception $e) {
      $this->loggerChannel->notice("Could not delete event %event_id : %error ", [
        '%event_id' => $event_id,
        '%error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }
  /**
   * Delete expired events.
   */
  public function deleteExpiredEvents() {
    $total = 0;
    $currentTime =  new DrupalDateTime()->getTimestamp();
    try {
      $total =  $this->database->delete($this->table)
        ->condition('expire', $currentTime, '<')
        ->execute();
    }
    catch (\Exception $e) {
      $this->loggerChannel->notice('Could not delete expired events : %error ', [
        '%error' => $e->getMessage(),
      ]);
    }
    if ($total){
      $this->loggerChannel->info('Removed @total expired events', ['@total' => $total]);
    }

  }

  public function queueEventsForProcessing(int $clients_per_run, int $min_events_per_client = 2): int {
    try {
      // Get the queue service
      $queue = \Drupal::queue('ga4_server_events_processor');
      $max_events_per_client = 25;
      $processed_clients = 0;


      $client_query = $this->database->select($this->table, 'events')
        ->fields('events', ['client_id'])
        ->condition('status', EventStatus::NEW->value)
        ->groupBy('client_id');

      $client_query->addExpression('COUNT(ulid)', 'event_count');
      $client_query->addExpression('MIN(created)', 'oldest_event');
      $client_query->having('COUNT(ulid) >= :min_events', [':min_events' => $min_events_per_client]);
      $client_query->orderBy('event_count', 'DESC');
      $client_query->orderBy('oldest_event');
      $client_query->range(0, $clients_per_run);
      $clients = $client_query->execute()->fetchCol();
      foreach ($clients as $client_id) {
        $query = $this->database->select($this->table, 'client_events')
          ->fields('client_events', ['ulid', 'client_id', 'event_type', 'event_timestamp', 'event_data', 'expire', 'created'])
          ->condition('client_id', $client_id)
          ->condition('status',EventStatus::NEW->value)
          ->orderBy('created', 'ASC')
          ->range(0, $max_events_per_client);

        $result = $query->execute();
        $events = [];
        $event_ids = [];

        foreach ($result as $record) {
          // Decode JSON data if it exists
          if (!empty($record->event_data)) {
            $record->event_data = json_decode($record->event_data, TRUE, 512, JSON_THROW_ON_ERROR);
          }
          $events[] = (array) $record;
          $event_ids[] = $record->ulid;
        }

        // If we found events for this client, queue them
        if (!empty($events)) {
          // Create a queue item for this client's events
          $queue->createItem([
            'client_id' => $client_id,
            'events' => $events,
            'event_ids' => $event_ids, // Keep track of IDs for later deletion or marking
            'timestamp' => new DrupalDateTime()->getTimestamp()
          ]);

          // Log that we're processing this client
          $this->loggerChannel->debug('Queued @count events for client @client_id', [
            '@count' => count($events),
            '@client_id' => $client_id,
          ]);

          $processed_clients++;
          $this->setEventStatus($event_ids, EventStatus::QUEUED);
        }

      }

      return $processed_clients;
    }
    catch (\Exception $e) {
      $this->loggerChannel->error('Error queueing events for processing: @error', [
        '@error' => $e->getMessage(),
      ]);
      return 0;
    }
  }

  /**
   * Get 100 single events so the can be flushed to GA
   */
  public function getSingleAboutToExpireEvents($max_clients = 100): array {
    $current_time = new DrupalDateTime()->getTimestamp();
    $one_hour_ago = $current_time- 3600;
    try {
      // First, create a subquery to get client_ids with exactly one event
      $subquery = $this->database->select($this->table, 'sub')
        ->fields('sub', ['client_id'])
        ->condition('status', EventStatus::NEW->value)
        ->condition('created', $one_hour_ago, '<')  // Created more than 1 hour ago
        ->groupBy('client_id');

      // Add the COUNT expression and having condition
      $subquery->addExpression('COUNT(ulid)', 'event_count');
      $subquery->havingCondition('event_count', 1, '=');


      // Get only the client_id column for the IN clause
      $client_ids_subquery = $this->database->select($subquery, 'filtered')
        ->fields('filtered', ['client_id']);

      // Main query to get event data for those clients
      $query = $this->database->select($this->table, 'events')
        ->fields('events', ['ulid', 'client_id', 'event_type', 'event_timestamp', 'event_data', 'expire', 'created'])
        ->condition('status', EventStatus::NEW->value)
        ->condition('created', $one_hour_ago, '<')
        ->condition('client_id', $client_ids_subquery, 'IN')
        ->range(0, $max_clients);
      // Execute the query
      $results = $query->execute();

      $events = [];
      if ($results){
        foreach ($results as $record) {
          // Decode JSON data if it exists
          if (!empty($record->event_data)) {
            $record->event_data = json_decode($record->event_data, TRUE, 512, JSON_THROW_ON_ERROR);
            $record->event_data['single_event'] = 1;
          }
          $events[] = [
            'ulid' => $record->ulid,
            'client_id' => $record->client_id,
            'events' => ['event_type' => $record->event_type,
              'event_timestamp' => $record->event_timestamp,
              'event_data' => $record->event_data,

            ],
            'expire' => $record->expire,
            'created' => $record->created,
          ];
        }
        return $events;
      }
    } catch (\Exception $e) {
      $this->loggerChannel->error('Error retrieving single events: @error', [
        '@error' => $e->getMessage(),
      ]);
      return [];
    }

  }

  public function setEventStatus(array $event_ids, EventStatus $status = EventStatus::NEW): void {
    try {
      // Get database connection service
      $database = \Drupal::database();

      // Update the events status
      $database->update('events_collector')
        ->fields(['status' => $status->value])
        ->condition('ulid', $event_ids, 'IN')
        ->execute();

      $this->loggerChannel->info('Updated @count events to status: @status', [
        '@count' => count($event_ids),
        '@status' => $status->label(),
      ]);
    }
    catch (\Exception $e) {
      $this->loggerChannel->error('Failed to update event status: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  public function getEventCountsByStatus(): array {
    try {
      $query = $this->database->select($this->table, 'e');
      $query->addExpression('COUNT(*)', 'count');
      $query->addField('e', 'status');
      $query->groupBy('status');

      $result = $query->execute()->fetchAllKeyed();

      // Ensure all statuses have a count, even if zero
      $counts = [];
      foreach (EventStatus::cases() as $status) {
        $counts[$status->value] = $result[$status->value] ?? 0;
      }

      return $counts;
    }
    catch (\Exception $e) {
      $this->loggerChannel->error('Failed to get event status counts: @error', [
        '@error' => $e->getMessage(),
      ]);

      // Return zero counts on error
      return array_fill_keys(
        array_map(fn($case) => $case->value, EventStatus::cases()),
        0
      );
    }
  }



}