<?php

declare(strict_types=1);

namespace Drupal\ga4_server\Client;

use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * Client for the Google Analytics 4 Measurement API.
 */
class MeasurementProtocol {

  /**
   * Standard GA4 Measurement API endpoint.
   */
  protected const string GA4_ENDPOINT = 'https://www.google-analytics.com/mp/collect';

  /**
   * Debug GA4 Measurement API endpoint.
   */
  protected const string GA4_DEBUG_ENDPOINT = 'https://www.google-analytics.com/debug/mp/collect';

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new EventsApiClient object.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    ClientInterface $httpClient,
    ConfigFactoryInterface $configFactory,
    LoggerInterface $logger
  ) {
    $this->httpClient = $httpClient;
    $this->configFactory = $configFactory;
    $this->logger = $logger;
  }

  /**
   * Sends GA4 events to the Measurement API.
   *
   * @param string $client_id
   *   The client ID to use for all events.
   * @param array $events
   *   The events to send (from the database).
   *
   * @return bool
   *   TRUE if all events were processed successfully, FALSE otherwise.
   */
  public function sendEvents(string $client_id, array $events): bool {
    $config = $this->configFactory->get('ga4_server.settings');
    $measurement_id = $config->get('measurement_id');
    $api_secret = $config->get('api_secret');

    if (empty($measurement_id) || empty($api_secret)) {
      $this->logger->error('GA4 Measurement ID or API secret is missing');
      return FALSE;
    }
    $debugMode = (bool) $config->get('debug_mode');

    // Build the complete URL with the measurement ID and API secret
    $url =  self::GA4_ENDPOINT . "?measurement_id={$measurement_id}&api_secret={$api_secret}";

    $success = TRUE;

    // Format the payload to match GA4 API requirements
    $payload = $this->formatPayload($client_id, $events);
    // Prepare request options - disable automatic exception throwing for HTTP errors
    $options = [
      'body' => $payload,
      'headers' => [
        'Content-Type' => 'application/json',
      ],
      'timeout' => 20,
      'connect_timeout' => 5,
    ];
    try {
      // Send the request
      $response = $this->httpClient->request('POST', $url, $options);

      if ($debugMode) {
        $this->logger->debug('Send to GA4 to @url with @options', [
          '@url' => $url,
          '@options' => $payload,
        ]);
      }

      $status_code = $response->getStatusCode();

      if ($status_code < 200 || $status_code >= 300) {
        $this->logger->error('GA4 API returned error status code: @code', [
          '@code' => $status_code,
        ]);
        $success = FALSE;
      }

      if ($debug_mode) {
        // In debug mode, GA4 returns validation results we should log
        $response_body = json_decode((string) $response->getBody(), TRUE, 512, JSON_THROW_ON_ERROR);
        if (!empty($response_body['validationMessages'])) {
          foreach ($response_body['validationMessages'] as $message) {
            $this->logger->warning('GA4 validation: @message', [
              '@message' => $message['description'] ?? 'Unknown validation message',
            ]);
          }
        }
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('HTTP connection error sending events to GA4: @error', [
        '@error' => $e->getMessage(),
      ]);
      $success = FALSE;
    }
    catch (\Throwable $e) {
      $this->logger->error('Error processing GA4 events: @error', [
        '@error' => $e->getMessage(),
      ]);
      $success = FALSE;
    }

    return $success;
  }

  /**
   * @throws \JsonException
   */
  protected function formatPayload(string $client_id, array $events): string {
    $ga4_events = [];

    foreach ($events as $event) {
      $event_data = $event['event_data'] ?? [];
      $ga4_event = [
        'name' => $event['event_type'],
        'params' => [],
      ];
      if (!empty($event['event_timestamp'])) {
        $ga4_event['params']['timestamp_micros'] = (int) $event['event_timestamp'];
      }

      if (is_array($event_data)) {
        foreach ($event_data as $key => $value) {
          if ($value === NULL || $value === '' || is_array($value) || is_object($value)) {
            continue;
          }
          if (preg_match('/^\w{1,40}$/', $key)) {
            $ga4_event['params'][$key] = $value;
          }
        }
      }
      $ga4_events[] = $ga4_event;
    }
    return json_encode([
      'client_id' => $client_id,
      'events' => $ga4_events,
      'non_personalized_ads' => TRUE,
    ], JSON_THROW_ON_ERROR);
  }

  /**
   * Validates GA4 Measurement API credentials.
   *
   * This can be used by the admin UI to check if the credentials work.
   *
   * @param string $measurement_id
   *   The GA4 Measurement ID to validate.
   * @param string $api_secret
   *   The GA4 API Secret to validate.
   *
   * @return bool
   *   TRUE if the credentials are valid, FALSE otherwise.
   */
  public function validateCredentials(string $measurement_id, string $api_secret): bool {
    // Use debug endpoint for validation
    $url = self::GA4_DEBUG_ENDPOINT . "?measurement_id={$measurement_id}&api_secret={$api_secret}";

    // Prepare a minimal test event
    $payload = [
      'client_id' => 'test-client-id',
      'events' => [
        [
          'name' => 'test_event',
          'params' => [
            'test_param' => 'test_value',
          ],
        ],
      ],
    ];

    try {
      $response = $this->httpClient->request('POST', $url, [
        'json' => $payload,
        'headers' => [
          'Content-Type' => 'application/json',
        ],
        'timeout' => 5,
        'connect_timeout' => 3,
        'http_errors' => FALSE, // Don't throw exceptions for HTTP errors
      ]);

      $status_code = $response->getStatusCode();

      // Check if the response indicates the credentials are valid
      if ($status_code >= 200 && $status_code < 300) {
        $response_body = json_decode((string) $response->getBody(), TRUE);

        // Even with 200 response, GA4 might return validation messages about the API Secret
        // Check if there's a message about invalid API secret
        return !(!empty($response_body['validationMessages'])
          && array_any($response_body['validationMessages'],
            static fn($message) => isset($message['fieldPath']) && $message['fieldPath'] === 'api_secret'));
      }

      return FALSE;
    }
    catch (\Throwable $e) {
      $this->logger->error('Error validating GA4 credentials: @error', [
        '@error' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

}