<?php

declare(strict_types=1);

namespace Drupal\ga4_server;


use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\ga4_server\Abstract\GaEventPlugin;
use Drupal\ga4_server\Attribute\GaEvent;
use Drupal\ga4_server\Database\EventsStorage;
use Symfony\Component\HttpFoundation\Session\Session;

/**
 * GaEvent plugin manager.
 */
final class GaEventPluginManager extends DefaultPluginManager {

  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * The session service.
   */
  protected Session $session;

  /**
   * The events storage service.
   */
  protected EventsStorage $eventStorage;

  /**
   * The language manager service.
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Constructs the object.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cache_backend,
    ModuleHandlerInterface $module_handler
  ) {
    parent::__construct('GaEvent', $namespaces, $module_handler, GaEventInterface::class, GaEvent::class);
    $this->alterInfo('ga_event_info');
    $this->setCacheBackend($cache_backend, 'ga_event_plugins');
  }

  /**
   * Sets the logger service.
   */
  public function setLogger(LoggerChannelInterface $logger) {
    $this->logger = $logger;
    return $this;
  }

  /**
   * Sets the session service.
   */
  public function setSession(Session $session) {
    $this->session = $session;
    return $this;
  }

  /**
   * Sets the events storage service.
   */
  public function setEventStorage(EventsStorage $eventStorage) {
    $this->eventStorage = $eventStorage;
    return $this;
  }

  /**
   * Sets the language manager service.
   */
  public function setLanguageManager(LanguageManagerInterface $languageManager) {
    $this->languageManager = $languageManager;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function createInstance($plugin_id, array $configuration = []) {
    $plugin_definition = $this->getDefinition($plugin_id);
    $plugin_class = $plugin_definition['class'];

    return new $plugin_class(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->logger,
      $this->session,
      $this->eventStorage,
      $this->languageManager
    );
  }

  public static function getPlugin(string $plugin_id): ?GaEventPlugin {
    $pluginManager = \Drupal::getContainer()->get('plugin.manager.ga_event');
    if ($pluginManager->hasDefinition($plugin_id)){
      return $pluginManager->createInstance($plugin_id);
    }
  }
}
