<?php

namespace Drupal\fact_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'fact_field_widget' widget.
 *
 * @FieldWidget(
 *   id = "fact_field_icon_widget",
 *   module = "fact_field",
 *   label = @Translation("Fact Field with icon selection list"),
 *   field_types = {
 *     "fact_field_type"
 *   }
 * )
 */
class FactFieldIconWidget extends FactFieldWidget {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'icon_mapping' => '',
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['fact_is_number'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Fact is a number'),
      '#default_value' => $this->getSetting('fact_is_number'),
    ];

    // If Fact is not a number
    $elements['fact_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Max fact length.'),
      '#default_value' => $this->getSetting('fact_size'),
      '#description' => $this->t('Maximum characters of the fact. Only applicable if the fact is not a number.'),
    ];

    $elements['fact_placeholder'] = [
      '#type' => 'textfield',
      '#title' => t('Placeholder'),
      '#default_value' => $this->getSetting('fact_placeholder'),
      '#description' => $this->t('Placeholder text of the fact field. Only applicable if the fact is not a number.'),
    ];


    // if amount is a number
    $elements['fact_max_value'] = [
      '#type' => 'number',
      '#title' => t('Amount max value'),
      '#default_value' => $this->getSetting('fact_max_value'),
      '#description' =>  $this->t('The maximum nummeric value of the fact field. Only applicable if the type is numeric.'),
    ];

    // Pre and postfix settings
    $elements['prefix_size'] = [
      '#type' => 'number',
      '#title' => t('prefix size'),
      '#default_value' => $this->getSetting('prefix_size'),
      '#description' =>  $this->t('Max length of the input field.'),
    ];

    $elements['postfix_size'] = [
      '#type' => 'number',
      '#title' => t('Postfix size'),
      '#default_value' => $this->getSetting('postfix_size'),
      '#description' =>  $this->t('Max length of the input field.'),
    ];

    $elements['icon_mapping'] = [
      '#type' => 'textarea',
      '#title' => t('Icon mapping'),
      '#default_value' => $this->getSetting('icon_mapping'),
      '#description' =>  $this->t('A list of possible icons in key|label format, 1 per line'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element['prefix'] = [
      '#title' => $this->t('Prefix'),
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->prefix ?? '',
      '#size' => $this->getSetting('prefix_size'),
    ];
    if ($this->getSetting('fact_is_number')) {
      $element['fact'] = [
        '#title' => $this->t('Fact'),
        '#type' => 'number',
        '#default_value' => $items[$delta]->fact ?? 1,
        '#max' => $this->getSetting('fact_max_value'),
      ];
    }
    else {
      $element['fact'] = [
        '#title' => $this->t('Fact'),
        '#type' => 'textfield',
        '#default_value' => $items[$delta]->fact ?? '',
        '#placeholder' => $this->getSetting('fact_placeholder'),
        '#size' => $this->getSetting('fact_size'),
        '#required' => TRUE,
      ];
    }

    $element['postfix'] = [
      '#title' => $this->t('Postfix'),
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->postfix ?? '',
      '#size' => $this->getSetting('postfix_size'),
    ];

    $icons = $this->getSetting('icon_mapping');
    if ($icons !== '') {
      $iconMap = $this->generateIconMap($icons);
      $element['extra'] = [
        '#title' => $this->t('Select icon'),
        '#type' => 'select',
        '#options' => $iconMap
      ];
    }
    return $element;
  }

  private function generateIconMap(mixed $icons): array {
    $icons = str_ireplace(["\r", "\n"], ['', '~'], $icons);
    $iconMap = explode('~', $icons);
    $result = [];
    foreach ($iconMap as $icon) {
      [$key, $value] = explode('|', $icon);
      $result[$key] = $value;
    }
    return $result;
  }
}