<?php

namespace Drupal\fact_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Plugin implementation of the 'fact_field_widget' widget.
 *
 * @FieldWidget(
 *   id = "fact_field_widget",
 *   module = "fact_field",
 *   label = @Translation("Fact Field"),
 *   field_types = {
 *     "fact_field_type"
 *   }
 * )
 */
class FactFieldWidget extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
        'fact_is_number' => TRUE,
        'fact_max_value' => 9999,
        'fact_placeholder' => '',
        'fact_size' => 100,
        'postfix_size' => 40,
        'prefix_size' => 40,
      ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['fact_is_number'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Fact is a number'),
      '#default_value' => $this->getSetting('fact_is_number'),
    ];

    // If Fact is not a number
    $elements['fact_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Max fact length.'),
      '#default_value' => $this->getSetting('fact_size'),
      '#description' => $this->t('Maximum characters of the fact. Only applicable if the fact is not a number.'),
    ];

    $elements['fact_placeholder'] = [
      '#type' => 'textfield',
      '#title' => t('Placeholder'),
      '#default_value' => $this->getSetting('fact_placeholder'),
      '#description' => $this->t('Placeholder text of the fact field. Only applicable if the fact is not a number.'),
    ];


    // if amount is a number
    $elements['fact_max_value'] = [
      '#type' => 'number',
      '#title' => t('Amount max value'),
      '#default_value' => $this->getSetting('fact_max_value'),
      '#description' =>  $this->t('The maximum nummeric value of the fact field. Only applicable if the type is numeric.'),
    ];

    // Pre and postfix settings
    $elements['prefix_size'] = [
      '#type' => 'number',
      '#title' => t('prefix size'),
      '#default_value' => $this->getSetting('prefix_size'),
      '#description' =>  $this->t('Max length of the input field.'),
    ];

    $elements['postfix_size'] = [
      '#type' => 'number',
      '#title' => t('Postfix size'),
      '#default_value' => $this->getSetting('postfix_size'),
      '#description' =>  $this->t('Max length of the input field.'),
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    if ($this->getSetting('fact_is_number')) {
      $summary[] = $this->t('Fact is a number of max : @value', ['@value' => $this->getSetting('fact_max_value')]);
    }
    else {
      $summary[] = $this->t('Fact is a string of max : @size characters', ['@size' => $this->getSetting('fact_size')]);
      $summary[] = $this->t('Placeholder : @placeholder', ['@placeholder' => $this->getSetting('fact_placeholder')]);
    }
    $summary[] = $this->t('Max length of Prefix / Postfix : @prefix / @postfix', [
      '@prefix' => $this->getSetting('prefix_size'),
      '@postfix' => $this->getSetting('postfix_size'),
    ]);
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element['prefix'] = [
      '#title' => $this->t('Prefix'),
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->prefix ?? '',
      '#size' => $this->getSetting('prefix_size'),
    ];
    if ($this->getSetting('fact_is_number')) {
      $element['value'] = [
        '#title' => $this->t('Fact'),
        '#type' => 'number',
        '#default_value' => $items[$delta]->value ?? 1,
        '#max' => $this->getSetting('fact_max_value'),
      ];
    }
    else {
      $element['value'] = [
        '#title' => $this->t('Fact'),
        '#type' => 'textfield',
        '#default_value' => $items[$delta]->value ?? '',
        '#placeholder' => $this->getSetting('fact_placeholder'),
        '#size' => $this->getSetting('fact_size'),
        '#required' => TRUE,
      ];
    }

    $element['postfix'] = [
      '#title' => $this->t('Postfix'),
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->postfix ?? '',
      '#size' => $this->getSetting('postfix_size'),
    ];


    return $element;
  }

}
