<?php

namespace Drupal\fact_field\Plugin\Field\FieldType;

use Drupal\Component\Utility\Random;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'fact_field_type' field type.
 *
 * @FieldType(
 *   id = "fact_field_type",
 *   label = @Translation("Fact Field"),
 *   description = @Translation("Provides a single number with a pre and post
 *   value"), default_widget = "fact_field_widget", default_formatter =
 *   "fact_field_values_formatter"
 * )
 */
class FactFieldType extends FieldItemBase {


  public static function defaultStorageSettings() {
    return [
        'prefix_max_length' => 40,
        'fact_max_length' => 100,
        'postfix_max_length' => 40,

      ] + parent::defaultStorageSettings();
  }

  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    $schema = [
      'columns' => [
        'prefix' => [
          'type' => 'text',
          'length' => (int) $field_definition->getSetting('prefix_max_length'),
          'not null' => FALSE,
        ],
        'fact' => [
          'type' => 'varchar',
          'length' => (int) $field_definition->getSetting('fact_max_length'),
          'not null' => TRUE,
        ],
        'postfix' => [
          'type' => 'varchar',
          'length' => (int) $field_definition->getSetting('postfix_max_length'),
          'not null' => FALSE,
        ],

      ],
    ];
    return $schema;
  }

  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    // Add our properties.
    $properties['prefix'] = DataDefinition::create('string')
      ->setLabel($this->t('Prefix'))
      ->setDescription($this->t('Prefix'));

    $properties['fact'] = DataDefinition::create('string')
      ->setLabel($this->t('Fact'))
      ->setDescription($this->t('Fact'))
      ->setRequired(TRUE);

    $properties['postfix'] = DataDefinition::create('string')
      ->setLabel($this->t('Postfix'))
      ->setDescription($this->t('Postfix'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function generateSampleValue(FieldDefinitionInterface $field_definition) {
    $random = new Random();
    $values['prefix'] = $random->word(random_int(1, $field_definition->getSetting('prefix_max_length')));
    $values['fact'] = $random->word(random_int(1, $field_definition->getSetting('fact_max_length')));
    $values['postfix'] = $random->word(random_int(1, $field_definition->getSetting('postfix_max_length')));
    return $values;
  }

  public function isEmpty() {
    $prefix = $this->get('prefix')->getValue();
    $fact = $this->get('value')->getValue();
    $postfix = $this->get('postfix')->getValue();
    return empty($prefix) && empty($fact) && empty($postfix);
  }

  /**
   * {@inheritdoc}
   */
  public function getConstraints() {
    $constraints = parent::getConstraints();
    $constraint_manager = \Drupal::typedDataManager()
      ->getValidationConstraintManager();
    $constraints[] = $constraint_manager->create('ComplexData', [
      'prefix' => [
        'Length' => [
          'max' => $this->getSetting('prefix_max_length'),
          'maxMessage' => t('%name: may not be longer than @max characters.', [
            '%name' => $this->getFieldDefinition()->getLabel(),
            '@max' => $this->getSetting('prefix_max_length'),
          ]),
        ],
      ],
      'fact' => [
        'Length' => [
          'max' => $this->getSetting('fact'),
          'maxMessage' => t('%name: can not be longer than @max characters.', [
            '%name' => $this->getFieldDefinition()->getLabel(),
            '@max' => $this->getSetting('fact_max_length'),
          ]),
        ],
      ],
      'postfix' => [
        'Length' => [
          'max' => $this->getSetting('postfix_max_length'),
          'maxMessage' => t('%name: can not be longer than @max characters.', [
            '%name' => $this->getFieldDefinition()->getLabel(),
            '@max' => $this->getSetting('postfix_max_length'),
          ]),
        ],
      ],
    ]);
    return $constraints;
  }

  /**
   * {@inheritdoc}
   */
  public function storageSettingsForm(array &$form, FormStateInterface $form_state, $has_data) {
    $elements = [];

    $elements['fact_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Fact length'),
      '#default_value' => $this->getSetting('fact_max_length'),
      '#required' => TRUE,
      '#description' => $this->t('The maximum length of the Fact.'),
      '#min' => 1,
      '#disabled' => $has_data,
    ];
    $elements['prefix_max_length'] = [
      '#type' => 'number',
      '#title' => $this->t('Prefix length'),
      '#default_value' => $this->getSetting('prefix_max_length'),
      '#required' => TRUE,
      '#description' => t('The maximum length of prefix.'),
      '#min' => 1,
      '#disabled' => $has_data,
    ];

    $elements['postfix_max_length'] = [
      '#type' => 'number',
      '#title' => t('Prefix length'),
      '#default_value' => $this->getSetting('postfix_max_length'),
      '#required' => TRUE,
      '#description' => t('The maximum length of postfix.'),
      '#min' => 1,
      '#disabled' => $has_data,
    ];

    return $elements;
  }

}
